C////////////////////////////////////////////////////////////////////////////////
C////
C////  R E S T R A X   4.8.1    EXCI
C////
C//// Subroutine called by RESTRAX to get values of excitation energy (OMEXC)
C//// and scattering cross-section (SQOM) for given QHKL,E values stored in Q(i)
C//// Permits to define up to 6 different branches of S(Q,E)
C////
C//// You can use this file as a template
C//// Refer to DON'T CHANGE .. END blocks for the code to be preserved
C////
C//// J. Saroun (saroun@ujf.cas.cz) , March 2005
C//// Read attached documentation or visit RESTRAX home page for help: 
C//// http://omega.ujf.cas.cz/restrax
C////////////////////////////////////////////////////////////////////////////////
C
C                           ***  ARGUMENTS ***
C input:
C Q(1:4)   ... (H,K,L,E) values 
C ICOM<-10 ... initialization (called only once when loaded at runtime )
C ICOM=0   ... initialization (run usually before each  [M]FIT or INIT  commands)
C ICOM=-1  ... only excitation energies are used (e.g. for plotting disp. branches)
C ICOM=-2  ... only S(Q,E) values are used (e.g. for plotting S(Q,E) maps)
C ICOM>0   ... should return both excitation energies and S(Q,E). ICOM=index of supplied event
C
C output:
C OMEXC(1:6) ... excitation energies for 1..nbr branches for Qhkl = Q(1:3) 
C SQOM(1:6)  ... S(Q,E) values for 1..nbr branches
C
C                           *** SHARED DATA ***
C
C Following fileds are available via common variables declared in the *.inc files:
C
C Monte Carlo ray-tracing results:
C-----------------------------------
C accessible only if ICOM>=0 !!
C REAL*4 QOM(1:4,j),PQOM(j) .... value of (Q,E) and weight for j-th event
C IQOM(j) .... the index of data set corresponding to given j-th event.
C NQOM(k) .... partitioning of the QOM, PQOM ... arrays, i.e. the number
C              of events stored for the k-th data set is NQOM(k)-NQOM(k-1).
C NDATQOM .... index of actual data set, for which the scan profile is accumulated
C              Use this index to define specific free parameters for different data sets 
C
C Instrument setting:
C-----------------------------------
C REAL*4 QOM0(1:4,k)   .... Spectrometer position (Q,E) for k-th data set
C
C Unit vectors in rec. lat. units:
C-----------------------------------
C REAL*8 PARAM(1:MPAR)         ... free model parameters
C INETEGR*4 FIXPARAM(1:MPAR)   ... fixed parameters. Set FIXPARAM(i)=0 to make  
C                                  the i-th parameter fixed)
C NTERM                        ... number of free model parameters (<=64)
C NBR                          ... number of branches defined by EXCI (<=6)
C REAL*8 WEN(1:6)              ... widths of the disp. branches. 
C CHARACTER*10 PARNAME(1:MPAR) ... names of free parameters
C                         
C Outside EXCI, WEN is used only as a flag to check, whether scattering 
C is difuse (WEN>0) or not (WEN=0). The convolution method is selected according 
C to this flag.  
C
C                           *** SHARED SUBROUTINES ***
C                        (see source files for details)
C   in this module:
C   SUBROUTINE READEXCIPAR     ... Read initial values of model variables 
C   in exci_io.f:
C   SUBROUTINE SETEXCIDEFAULT  ... Set default values to common EXCI variables 
C   in reclat.f:
C   SUBROUTINE POLVECT(Q,TAU,SIG1,SIG2,SIG3,ICOM) ... Get polarization unit vectors with 
C                                                     respect to q=TAU-Q 
C   REAL*8 FUNCTION QxQ(A,B)   ... Scalar product of vectors A,B in non-carthesian rec. lattice coordinates
C   SUBROUTINE QNORM(X,QRLU,QANG)   ... Norm of a vector X in non-carthesian rec. lattice coordinates
C///////////////////////////////////////////////////////////////////////////////////
C------------------------------------------------------------------------------
      SUBROUTINE EXCI(ICOM,Q,OMEXC,SQOM)
C Bond charge model: phonons in diamond lattice (Si, Ge, ...)
C     REFERENCE: W.WEBER, PHYS.REV.B,VOL.15,NO.10,(1977),4789.
C                O.H.NIELSEN, W.WEBER, COMPUTER PHYSICS COMM.(1979)
C The energy profile S(E) is either zero-width [wen(j)=0], or Lorenzian [wen(j)>0]
C------------------------------------------------------------------------------
      IMPLICIT NONE
      
C----------------------- *** DON'T CHANGE *** ------------------------------
      INCLUDE 'const.inc'
c      INCLUDE 'inout.inc'
      INCLUDE 'exci.inc'
      INTEGER*4 ICOM,EXCINIT
      REAL*8 Q(4),OMEXC(6),SQOM(6)
      
C-------------------------- *** END *** ------------------------------------
      
C **** Local user declarations ****

      REAL*8 pihalf,tau(3)
      PARAMETER (pihalf=1.570796327D0)   
      CHARACTER*16 CNAME(6)
      integer*4 j,i,k,nptmc,nlog
      real*8 bf,kT,z,arg
      complex eivec(MQOM,6,6),eivbmc(6,6),f(6),phfac
      real*4  qphon(3),omegaint(MQOM,6),ombmc(6)!,phase
      real*8 lastCHK
      integer*4 lastimod
      SAVE eivec,omegaint
      data tau/2.D0,2.D0,0.D0/
C// internal variables describing the model
C// This common is not shared with the rest of RESTRAX, only within this file. 
      integer*4 imodel
      real*8 temp,fqsq(6),omexc0(6),omscal
      common /excipar/ CNAME,temp,fqsq,omexc0,omscal,imodel
C// for debugging only, use with bcm_debug.f:      
c      real*4 q2(3),om2(6)
c      real*8 f2(6)
c      complex cvec(6),eiv2(6,6)
      
      
C------------------- *** DATA section *** ----------------

C **** DEFAULT values of internal model variables ****
      DATA lastCHK/1.1D0/ 
      DATA lastimod/-1/
      DATA CNAME /'Diamond','Silicon','Germanium','alpha-Tin','Ge70',
     &           'Ge76'/     

      DATA temp/300.D0/
      DATA fqsq/6*1.D0/
c      DATA tau/1.,0.,0./tauread/0/
      DATA omscal/1.0/
      DATA omexc0/5*0.0,5.0/
      DATA EXCINIT/0/

C***********************************************************************************
C MODEL INITIALIZATION (ICOM<-10)
C***********************************************************************************
C-- called only once when loaded at runtime 
C-- set some values shared with RESTRAX if different from default

      IF (ICOM.LT.-10) THEN

C Set model identification string: 
        PHONTITLE=
     * 'Bond charge model: phonons in diamond lattice (Si, Ge, ...)' 
              
C// Define fixed parameters (=0), default: all free (=1)
c        FIXPARAM(1)=0  ! let Intensity fixed !!

C// Number of branches ****    
        NBR=6

C// Initial widths in energy, default=1meV 
C// Set wen(i)=0 for zero-width branches
        DO I=1,6
          WEN(1)=0.D0 
        ENDDO  

C**** How to read file with parameters (default=1):
C**** (0) never (1) at program start or on INIT command (2) each time MFIT is called   
c        EXCREAD=0 
      
C Set name of file with model parameters (if different from default exc.par)
        PHONNAME='bcm.par'
      
C Define names of free parameters for i>2:
        PARNAME(3)='EN_scale'
        PARNAME(4)='W_LO'       
        PARNAME(5)='W_TO1'       
        PARNAME(6)='W_TO2'       
        PARNAME(7)='W_LA'       
        PARNAME(8)='W_TA1'       
        PARNAME(9)='W_TA2'          

C// number of free model parameters
        NTERM=9
        
        write(*,*) 'EXCI: set default'
        return
      ENDIF
 
C----------------------- *** DON'T CHANGE *** ------------------------------  
      IF ((ICOM.NE.0).AND.(EXCINIT.NE.0)) GOTO 1      
C---------------------------- *** END *** ----------------------------------

C***********************************************************************************
C MODEL INITIALIZATION (ICOM=0)                    
C***********************************************************************************
C-- called before each [M]FIT or INIT command  
C

c *** initialize BCM
      call parbcm(imodel,0)      

      nptmc=NQOM(MDAT) ! get number of events stored for all data sets
      
c *** tabulate BCM if model or QOM values changed
      if (lastimod.NE.imodel.or.lastCHK.ne.CHKQOM) then
10    format('Tabulating BCM frequencies & eigenvectors for ',I8,' points',$)
11    format('.',$)
      write(*,10) nptmc
      nlog=nptmc/20
      do i=1,nptmc
        if (nlog.gt.0.and.mod(i,nlog).EQ.0) write(*,11)
        do j=1,3
          qphon(j) = qom(j,i) !-tau(j)
        enddo
        call bcmres(qphon,ombmc,eivbmc)
        do k=1,6
          omegaint(i,k) = ombmc(k)
          do j=1,6
            eivec(i,j,k) = eivbmc(j,k) !k ... branch, j ... comp.
          enddo
        enddo
      enddo
      write(*,*)'finished.'
      lastCHK=CHKQOM
      lastimod=imodel
      endif

C// Assign free model parameters to param():
      param(3)=omscal
      do i=1,NBR 
        param(3+i)= wen(i) 
      enddo  
      
C----------------------- *** DON'T CHANGE *** -------------------------

      EXCINIT=1
      RETURN
C---------------------------- *** END *** ----------------------------------
1     CONTINUE      

C********************************************************************************
C                                                                   
C                   EXECUTION PART (ICOM<>0)                        
C
C This part is called many times during the fitting procedure 
C =>  should be as fast as possible
C 
C// Do whatever you want in the following code. 
C// EXCI MUST RETURN: 
C// OMEXC(i) ... excitation energies for first NBR branches (i=1..6)
C// SQOM(i)  ... dS/dOmega/dE 


C ICOM=-1 => only OMEXC(i) values are used by RESTRAX to plot the branches.
C ICOM=-2 => both OMEXC and SQOM are needed, but no data set is provided (used for mapping S(Q,E)
C Otherwise, ICOM refers to the event number in the QOM array
C   => ICOM can be used e.g. as an index to internal lookup tables of EXCI etc...                                
C********************************************************************************
      

C------------------!! OBLIGATORY !!-------------------------

C// Assign values in the PARAM array to the local model variables
C// if you don't work with the PARAM() array directly
C// REMEMBER: PARAM(1,2) are reserved for Scale and Background

      omscal = param(3)
      do j=1,nbr
        wen(j) = param(3+j)
      enddo

C// don't allow wen->0, if it is not a zero-width branch  !!      
      DO i=1,NBR
         wen(i)=abs(wen(i))
         if(wen(i).ne.0.D0.AND.wen(i).lt.1e-3) wen(i) = 1.e-3 
      ENDDO     
        
C----------------------!! END !! -------------------------

C*  If ICOM=-1, return only excitation energies
      IF(icom.eq.-1) then
        do j=1,3
          qphon(j) = q(j) !-tau(j)
        enddo
        call bcmres(qphon,ombmc,eivbmc)
        do j=1,6
          omexc(j) = omscal*ombmc(j)
          sqom(j) = 1.
       enddo
       RETURN
      ENDIF

C///  Returns up to six energies and cross-sections:
         
      do i=1,3
         qphon(i) = q(i) !-tau(i)  ! get actual phonon q
      enddo
      
C/// get energies and eigenvectors from table if possible
C/// otherwise call bcmres

C// check the difference between tabulated and actual Q-value      
      Z=0.D0
      if (icom.gt.0) then
        do j=1,3
         Z=Z+ABS(qom(j,icom)-q(j))
        enddo 
      endif
        
C// calculate eigenvalues/vectors if qom differes from initialization input or if icom<0
      if (icom.le.0.or.z.gt.1e-5) then
        call bcmres(qphon,ombmc,eivbmc)      
      else
C// otherwise use values calculated during initialization
        do k=1,nbr
          ombmc(k) = omegaint(icom,k)
          do j=1,6
            eivbmc(j,k)=eivec(icom,j,k)  !k ... branch, j ... comp.
          enddo
        enddo
      endif   

C// scale energies               
      do k=1,nbr
        omexc(k) = omscal*ombmc(k)
      enddo
 
c *** calculate cross-section

c      phase = pihalf*(tau(1)+tau(2)+tau(3))
c      phfac = cexp(cmplx(0.D0,phase))
      phfac=cmplx(1.0,0.0)
      do k=1,nbr
        f(k) = cmplx(0,0)
          do j=1,3
            f(k)=f(k)+q(j)*(eivbmc(j,k)*phfac+eivbmc(j+3,k))
          enddo
        fqsq(k)=(abs(f(k)))**2
      enddo


C This was for debugging  only, comment the following include out for normal use
c-----------------------------------------------------------------------
c      INCLUDE 'bcm_debug.f'  
c-----------------------------------------------------------------------
      
           
100   kT=temp/11.609 ! 11.609 ... conversion kT -> meV
      do k=1,nbr
c// bf = Bose factor x omega
        z=exp(-omexc(k)/kT)  
        if (abs(1.0-z).LT.1.D-5) then
          bf=kT
        else
          bf=omexc(k)/(1-z)
        endif          
        if(fqsq(k).eq.0) then
          sqom(k)=0.
        else
          if(abs(omexc(k)).LT.0.0001) omexc(k)=0.0001
          if(wen(k).gt.0.) then ! apply finite width
c             arg = 2.*(omexc(k)-q(4))/wen(k)
c             sqom(k) = bf*fqsq(k)*2./(wen(k)*(1.+arg**2))
c modified: use damped oscillators instead of Lorentz dist.          
            arg=((q(4)**2-omexc(k)**2)**2+wen(k)**2*q(4)**2)/wen(k)
            if(abs(arg).LT.1.E-8) arg=1.e-8
            sqom(k) = bf*fqsq(k)/abs(arg)
           else
             if(abs(omexc(k)).LT.1.E-4) omexc(k)=1.e-4
             sqom(k) = bf*fqsq(k)/omexc(k)**2
           endif
        endif
      enddo

      END
      
C------------------------------------------------------------------------------
      SUBROUTINE REPEXCIPAR
C REPORT model ID and input parameters as needed
C------------------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
c      INCLUDE 'inout.inc'
      INCLUDE 'exci.inc'
      
      CHARACTER*16 CNAME(6)
      integer*4 imodel
      real*8 temp,fqsq(6),omexc0(6),omscal
      common /excipar/ CNAME,temp,fqsq,omexc0,omscal,imodel
      
      write(*,*) 'EXCI: ',trim(PHONTITLE)
C// Report some model values:  
      write(*,*) 'Material: ',trim(CNAME(imodel))
12    format(' Temperature [K]: ',G10.4)
      write(*,12) temp
     
      END
      
C------------------------------------------------------------------------------
      SUBROUTINE READEXCIPAR
C Read values of model variables used by EXCI
C Call by RESTRAX when requiared
C File is opened and closed by RESTRAX, don't call OPEN/CLOSE here !!!
C------------------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
c      INCLUDE 'inout.inc'
      INCLUDE 'exci.inc'
     

      CHARACTER*16 CNAME(6)
      integer*4 imodel
      real*8 temp,fqsq(6),omexc0(6),omscal
      common /excipar/ CNAME,temp,fqsq,omexc0,omscal,imodel
      INTEGER*4 i
           
      rewind(EXCUNIT) ! call rewind for compatibility with g77
C read some model parameters from file 
      read (EXCUNIT,*,err=998) temp  ! temperature     
      if (temp.le.0.) temp=0.01

      read (EXCUNIT,*,err=998) imodel  ! Diamond, Si, Ge, ...
      if (imodel.gt.6.or.imodel.le.0) imodel=1
      read (EXCUNIT,*,err=30) (wen(i),i=1,6) ! # widths (optional)              
            
30    continue
      write(*,*) 'Parameters updated from '//PHONNAME
      RETURN
      
998   write(*,*) 'Format error?! Cannot read excitation parameters.'
      RETURN
      END
      
      
C--------------------------------------------------------------------
      SUBROUTINE BCMRES (Q,FRMEV,VEC)
C     Bond charge model - modified by J.K. Oct-1998
C ***	- uses real*4 arithmetics everywhere
C ***  procedure for Restrax calls
C *** PARAM is called PARBCM to avoid confusion
C--------------------------------------------------------------------

      REAL Q(3),FRMEV(6),WR(6,6),WI(6,6)
      COMPLEX VEC(6,6)
C
C     BOND CHARGE MODEL IN DIAMOND-SYMMETRY
C     REFERENCE: W.WEBER, PHYS.REV.B,VOL.15,NO.10,(1977),4789.
C                O.H.NIELSEN, W.WEBER, COMPUTER PHYSICS COMM.(1979)
C
C     INPUT:
C
C     Q ... WAVEVECTOR
C           WE DEFINE THE 1BZ BY:
C           X=(1,0,0)
C           L=(.5,.5,.5)
C
C     OUTPUT:
C
C     FRTHZ ...  ARRAY CONTAINING THE EIGENVALUES IN INCREASING ORDER
C     WR,WI ...  REAL & IMAGINARY PART OF THE CORRESPONDING EIGENVECTORS
C                FOR FRTHZ(I) THE EIGENVECTOR IS:
C                W(J,I),J=1..6
C
C                COMMON /PRAM/  from param.fortran
C
C     FCBC ..... BOND CHARGE FORCE CONSTANTS, USING NOTATION OF W.WEBER:
C                1-ALFA; 2-BETA; 3-MU; 4-NU; 5-LAMBDA; 6-DELTA
C     FCION .... ION-ION FORCE CONSTANTS, SAME AS FCBC, BUT WITH A PRIME
C                IN W.WEBER'S NOTATION.
C     MASS ..... MASS OF THE ATOM.
C     SCALE .... A SCALE FACTOR TO CONVERT FREQUENCIES INTO THZ, SINCE
C                INTERNALLY WE USE FORCE CONSTANT UNITS E**2/VA.
C     ACELL2 ... 1/2 OF CELL LENGTH, DENOTED R0 IN W.WEBER'S ARTICLE.
C     Z2EPS .... Z**2/EPSILON (MODEL PARAMETER).
C
C     COMMON AREA OUTPUT, WHICH THE USER MIGHT WISH TO USE:
C
C     EXPRAM ... EXPERIMENTAL RAMAN FREQUENCY.
C     THRAM .... THEORETICAL RAMAN FREQUENCY.
C     CELAST ... ELASTIC CONSTANTS ARRANGED AS:
C                C11  C12  C44  EXPERIMENTAL
C                C11  C12  C44  CALCULATED
C
      COMMON /PRAM/ FCBC(6),FCION(6),MASS,SCALE,ACELL2,Z2EPS,EXPRAM,THR
     1AM,CELAST(3,2)
      REAL MASS
C
C     QPI ... WAVEVECTOR MULTIPLIED BY PI
C
      COMMON /QVECT/ QPI(3)
C
C     A ... PRIMITIVE TRANSLATIONAL VECTORS.
C           A(I,J) IS COORDINATE I OF THE J'TH VECTOR.
C     R12 . CONTAINS ALL THE RELATIVE COORDINATES OF THE IONS AND
C           BOND CHARGES, I.E. X(K')-X(K).
C           R12(I,J) IS COORDINATE J OF RELATIVE VECTOR I.
C
      COMMON /BASIS/ A(3,3),R12(12,3)
C
      REAL CRR(6,6),CRI(6,6),CTR(6,12),CTI(6,12),CS(12,12),TR(6,12),TI(6
     1,12),S(12,12),UR(12,6),UI(12,6),TST1(6,6),TST2(6,6),TSTR(6,6),TSTI
     2(6,6),RR(6,6),RI(6,6)
              
     
      DATA PI/3.1415926535898/
      data clight,mev/33.3564095198,4.135701/
c
C     AT GAMMA, WE CHOOSE Q TO LIE ALONG (100)
      IF (ABS(Q(1))+ABS(Q(2))+ABS(Q(3)).LE.1.0E-6) Q(1)=0.00001
C
      DO I=1,3
        QPI(I)=Q(I)*PI      !     WE NEED THE WAVEVECTOR TIMES PI
      END DO
C
C     CONSTRUCT THE COULOMB MATRICES CR,CT,CS
      CALL CCMAT (CRR,CRI,CTR,CTI,CS)
C
      CALL MATOP ('.    ',CRR,6,6,CRR,6,6,CRR,Z2EPS)
      CALL MATOP ('.    ',CRI,6,6,CRI,6,6,CRI,Z2EPS)
      CALL MATOP ('.    ',CTR,6,12,CTR,6,12,CTR,Z2EPS)
      CALL MATOP ('.    ',CTI,6,12,CTI,6,12,CTI,Z2EPS)
      CALL MATOP ('.    ',CS,12,12,CS,12,12,CS,Z2EPS)
C
C     CONSTRUCT MATRIX T
      CALL TMAT (TR,TI)
      CALL MATOP ('+    ',TR,6,12,CTR,6,12,TR,1.0)
      CALL MATOP ('+    ',TI,6,12,CTI,6,12,TI,1.0)
      CALL MATOP ('TRANS',TR,6,12,UR,12,6,TR,1.0)
      CALL MATOP ('TRANS',TI,6,12,UI,12,6,TI,-1.0)
C
C     CONSTRUCT MATRIX S
      CALL SMATX (S)
C
      CALL MATOP ('+    ',S,12,12,CS,12,12,S,1.0)
      CALL MATOP ('INV  ',S,12,12,CS,12,12,CS,1.0)
      CALL MATOP ('*    ',TR,6,12,CS,12,12,CTR,1.0)
      CALL MATOP ('*    ',CTR,6,12,UR,12,6,TST1,1.0)
      CALL MATOP ('*    ',TI,6,12,CS,12,12,CTI,1.0)
      CALL MATOP ('*    ',CTI,6,12,UI,12,6,TST2,1.0)
      CALL MATOP ('-    ',TST1,6,6,TST2,6,6,TSTR,1.0)
      CALL MATOP ('*    ',CTI,6,12,UR,12,6,TST1,1.0)
      CALL MATOP ('*    ',CTR,6,12,UI,12,6,TST2,1.0)
      CALL MATOP ('+    ',TST1,6,6,TST2,6,6,TSTI,1.0)
C
C     CONSTRUCT MATRIX R
      CALL RMAT (RR,RI)
C
      CALL MATOP ('+    ',RR,6,6,CRR,6,6,RR,1.0)
      CALL MATOP ('+    ',RI,6,6,CRI,6,6,RI,1.0)
      CALL MATOP ('-    ',RR,6,6,TSTR,6,6,RR,1.0)
      CALL MATOP ('-    ',RI,6,6,TSTI,6,6,RI,1.0)
C
C     THE TOTAL DYNAMICAL MATRIX: (WR,WI)
C
C     WEIGHT WITH MASS FACTORS
      CALL MATOP ('.    ',RR,6,6,WR,6,6,WR,1.0/MASS)
      CALL MATOP ('.    ',RI,6,6,WI,6,6,WI,1.0/MASS)

c check eigenvalues
c10    format(a,6(2x,G14.8))
c11    format(a,6(2x,G10.4))
c      write(*,10) 'Q: ',(Q(I),I=1,3)
c      write(*,10) 'WI:',(WI(I,I),I=1,6)
c      do i=1,6
c      do j=1,6
c         crr(i,j)=WR(i,j)
c	 cri(i,j)=WI(i,j)
c      enddo
c      enddo

C
C     SET THE (HOPEFULLY) SMALL DIAGONAL IMAGINARY ELEMENTS = 0
C

      DO 120 I=1,6
      WI(I,I)=0.0
120   CONTINUE
C
C     SOLVE THE EIGENVALUE PROBLEM
C

      CALL MATOP ('EIG  ',WR,6,6,WI,6,6,TST1,1.0)

      
c      write(*,10) 'TESTING EISPACK: ',ierr*1.D0

c      do j=1,6
c        sum=0.
c	do i=1,6
c	  sum=sum+wr(i,j)**2+wi(i,j)**2
c	enddo
        
c        write(*,11) 'OM, NORM(e): ',tst1(j,j),sqrt(sum)
c      do i=1,6
c        tstr(i,j)=0.
c	tsti(i,j)=0.
c	do k=1,6
c	  tstr(i,j)=tstr(i,j)+crr(i,k)*wr(k,j)-cri(i,k)*wi(k,j)
c	  tsti(i,j)=tsti(i,j)+crr(i,k)*wi(k,j)+cri(i,k)*wr(k,j)
c	enddo  
c	write(*,11) 'M*EIG, OM*EIG, EIG:', tstr(i,j),tst1(j,j)*wr(i,j),
c     &         wr(i,j),wi(i,j)
c      enddo
c      enddo
c      pause
      
      
C
C     TAKE "SQUARE ROOT"
C
      DO 130 I=1,6
      FRMEV(I)=SQRT(ABS(TST1(I,I)*SCALE))*SIGN(1.0,TST1(I,I))*mev
130   CONTINUE
c
c *** the eigenvectors are in C-convention
c
c *** to convert to Strauch's-convention
c *** (include trivial phase factor) uncomment the following ***
c
c     phase = .5*(qpi(1)+qpi(2)+qpi(3))
c      cph = cos(phase)
c      sph = sin(phase)
c      do j=1,6
c        do i=4,6
c          wwr =wr(i,j)*cph-wi(i,j)*sph
c          wwi =wi(i,j)*cph+wr(i,j)*sph
c          wr(i,j) = wwr
c          wi(i,j) = wwi
c        enddo
c      enddo
c *** down to here *********************************************
C
      DO I=1,6           ! added by STZO
        DO J=1,6
          VEC(I,J)=CMPLX(WR(I,J),WI(I,J))
        END DO
      END DO
C
      RETURN
      END

C
C--------------------------------------------------------------------
      SUBROUTINE CCMAT (CRR,CRI,CTR,CTI,CS)
      REAL CRR(6,6),CRI(6,6),CTR(6,12),CTI(6,12),CS(12,12)
C
C     CONSTRUCT THE COULOMB MATRICES BY CALLING COULCF
C
C     OUTPUT:
C     CRR,CRI ... REAL & IMAG PART OF MATRIX CR
C     CTR,CTI ... REAL & IMAG PART OF MATRIX CT
C     CS ........ MATRIX CS (A REAL MATRIX BY SYMMETRY)
C
C--------------------------------------------------------------------
      COMPLEX BBB(3,3)
      REAL AA(6,6),AB(6,6),BA(6,6),BB(6,6),R(3),BBBR(12,3,3),BBBI(12,3,3
     1),A1(3,3),A2(3,3),A3(3,3),A4(3,3)
      COMMON /QVECT/ QPI(3)
      COMMON /BASIS/ A(3,3),R12(12,3)
      DATA EPS/1.3/,ICOUNT/0/,TP/83.99769/
      SAVE A1,A2,A3,A4 ! static variables, added by J.S. 
C
C     CALL GENERAL COULOMB COEFFICIENT ROUTINE
C
      DO 110 I=1,12
      DO 100 J=1,3
      R(J)=R12(I,J)
100   CONTINUE
      CALL COULCF (BBB,QPI,A,R,EPS)
C
      DO 110 J=1,3
      DO 110 K=1,3
      BBBR(I,J,K)=REAL(BBB(J,K))
      BBBI(I,J,K)=AIMAG(BBB(J,K))
110   CONTINUE
911   format(12f6.2)
C
C     GENERATE MATRIX CR
C
      DO 120 I=1,3
      DO 120 J=1,3
      I3=I+3
      J3=J+3
      CRR(I,J)=4.0*BBBR(1,I,J)
      CRR(I3,J3)=CRR(I,J)
      CRI(I,J)=4.0*BBBI(1,I,J)
      CRI(I3,J3)=CRI(I,J)
      CRR(I,J3)=4.0*BBBR(2,I,J)
      CRR(I3,J)=CRR(I,J3)
      CRI(I,J3)=4.0*BBBI(2,I,J)
      CRI(I3,J)=-CRI(I,J3)
120   CONTINUE
C
C     GENERATE MATRIX CT
C
      DO 140 I=1,3
      DO 130 J=1,3
      CTR(I,J)=-2.0*BBBR(3,I,J)
      CTI(I,J)=-2.0*BBBI(3,I,J)
      CTR(I,J+3)=-2.0*BBBR(4,I,J)
      CTI(I,J+3)=-2.0*BBBI(4,I,J)
      CTR(I,J+6)=-2.0*BBBR(5,I,J)
      CTI(I,J+6)=-2.0*BBBI(5,I,J)
      CTR(I,J+9)=-2.0*BBBR(6,I,J)
      CTI(I,J+9)=-2.0*BBBI(6,I,J)
130   CONTINUE
      DO 140 J=1,12
      CTR(I+3,J)=CTR(I,J)
      CTI(I+3,J)=-CTI(I,J)
140   CONTINUE
C
C     GENERATE MATRIX CS
C
C     FIRST TIME CALLING CCMAT, WE BUILD THE MATRICES A1..A4
      IF (ICOUNT.EQ.1) GO TO 150
      ICOUNT=1
      CALL MAT33 (0.,-1.,-1.,-1.,0.,-1.,-1.,-1.,0.,TP,A1)
      CALL MAT33 (0.,1.,-1.,1.,0.,1.,-1.,1.,0.,TP,A2)
      CALL MAT33 (0.,1.,1.,1.,0.,-1.,1.,-1.,0.,TP,A3)
      CALL MAT33 (0.,-1.,1.,-1.,0.,1.,1.,1.,0.,TP,A4)
150   DO 160 I=1,3
      DO 160 J=1,3
      I3=I+3
      J3=J+3
      AA(I,J)=BBBR(1,I,J)+A1(I,J)
      AA(I3,J3)=BBBR(1,I,J)+A2(I,J)
      AA(I,J3)=BBBR(7,I,J)
      AA(I3,J)=AA(I,J3)
      BB(I,J)=BBBR(1,I,J)+A3(I,J)
      BB(I3,J3)=BBBR(1,I,J)+A4(I,J)
      BB(I,J3)=BBBR(12,I,J)
      BB(I3,J)=BB(I,J3)
      AB(I,J)=BBBR(8,I,J)
      BA(I,J)=AB(I,J)
      AB(I,J3)=BBBR(9,I,J)
      BA(I3,J)=AB(I,J3)
      AB(I3,J)=BBBR(10,I,J)
      BA(I,J3)=AB(I3,J)
      AB(I3,J3)=BBBR(11,I,J)
      BA(I3,J3)=AB(I3,J3)
160   CONTINUE
      CALL COUPL (AA,AB,BA,BB,CS,6,6,12,12)
      RETURN
      END

C--------------------------------------------------------------------
      SUBROUTINE COULCF (BBB,QPI,A,XK,EPS)
C
C     GENERAL CALCULATION OF
C     COULOMB-COEFFICIENTS BY THE EWALD METHOD
C     REF: MARADUDIN ET.AL., SOLID STATE PHYSICS, SUPP.3, 2ND.ED.,
C          "LATTICE DYNAMICS IN THE HARMONIC APPROXIMATION", CH.6.2
C          USING THE FORMULAS (6.2.18,21,22,24,53,64)
C
C     BBB ... COULOMB COEFFICIENT MATRIX
C             INCLUDING THE IRREGULAR PART|
C     QPI ... WAVEVECTOR MULTIPLIED BY PI.
C     A ..... TRANSLATIONAL BASIS VECTORS
C             1ST INDEX ARE COORDINATES, 2ND ARE ATOM #
C     XK  ... THE DISPLACEMENT VECTOR X(K')-X(K)
C     EPS ... CONVERGENCE PARAMETER FOR WIDTH OF GAUSSIAN
C             SEE MARADUDIN P.204
C
C--------------------------------------------------------------------
      REAL*4 erff
      REAL QPI(3), A(3,3), XK(3)
      COMPLEX BBB(3,3)
      DIMENSION B(3,3),KRM(3),KDM(3),TAUK(3),XL(3),XLK(3),HAB(3,3)
      COMPLEX PHFAC
      LOGICAL RZERO
      DATA PI,PI2,PI4,PISQR/3.1415926535898,6.2831853072,12.5663706144,1
     1.7724538509/
C     CONVERGENCE PARAMETER FOR ACCURACY ABOUT 10**(-8)
      DATA CONVRG/4.17/
C
C     RECIPROCAL SPACE BASIS, VOLUME OF UNIT CELL
C
      DO 100 I=1,3
      I1=MOD(I,3)+1
      I2=MOD(I+1,3)+1
      DO 100 J=1,3
      J1=MOD(J,3)+1
      J2=MOD(J+1,3)+1
      B(I,J)=A(J1,I1)*A(J2,I2)-A(J1,I2)*A(J2,I1)
100   CONTINUE
      VA=A(1,1)*B(1,1)+A(1,2)*B(2,1)+A(1,3)*B(3,1)
      DO 110 I=1,3
      DO 110 J=1,3
      B(I,J)=B(I,J)/VA
110   CONTINUE
      VA=ABS(VA)
C
C     CALCULATION OF PARAMETERS
C
C     SQRP**2 IS THE WIDTH OF THE GAUSSIAN (SEE MARADUDIN P.204)
      SQRP=EPS/(VA**.33333333)
      P4=4.0*SQRP**2
      PVA=EPS**3
      RZERO=(XK(1)**2+XK(2)**2+XK(3)**2).LT.1.0E-10
C     AT QPI = 0 WE CHOOSE TO GO ALONG (100)-DIRECTION
      IF (QPI(1)**2+QPI(2)**2+QPI(3)**2.LT.1.0E-10) QPI(1)=0.00001
      DO 120 I=1,3
      DO 120 J=1,3
      BBB(I,J)=(0.0,0.0)
120   CONTINUE
C
C     CALCULATION OF MAXIMAL LATTICE VECTORS
C
      DO 130 I=1,3
      KDM(I)=IFIX(CONVRG/SQRP*SQRT(B(I,1)**2+B(I,2)**2+B(I,3)**2)+0.5)
      KRM(I)=IFIX(CONVRG*SQRP/PI*SQRT(A(1,I)**2+A(2,I)**2+A(3,I)**2)+0.5
     1)
130   CONTINUE
C
C     SUMMATION IN RECIPROCAL SPACE
C     (MARADUDIN EQ.6.2.18)
C
      KRM1=2*KRM(1)+1
      KRM2=2*KRM(2)+1
      KRM3=2*KRM(3)+1
      DO 160 K1=1,KRM1
      FK1=FLOAT(K1-KRM(1)-1)
      DO 160 K2=1,KRM2
      FK2=FLOAT(K2-KRM(2)-1)
      DO 160 K3=1,KRM3
      FK3=FLOAT(K3-KRM(3)-1)
      DO 140 J=1,3
C     TAU + K
      TAUK(J)=PI2*(FK1*B(1,J)+FK2*B(2,J)+FK3*B(3,J))+QPI(J)
140   CONTINUE
C     (TAU + K)*XK
      TAUKR=TAUK(1)*XK(1)+TAUK(2)*XK(2)+TAUK(3)*XK(3)
C     (TAU + K)**2
      TAUK2=TAUK(1)**2+TAUK(2)**2+TAUK(3)**2
      TAUK2E=TAUK2/P4
      IF (TAUK2E.GT.CONVRG**2.OR.TAUK2E.EQ.0.0) GO TO 160
      PHFAC=CEXP(CMPLX(-TAUK2E,-TAUKR))
      DO 150 I=1,3
      DO 150 J=1,3
      BBB(I,J)=BBB(I,J)+CMPLX(TAUK(I)*TAUK(J)/TAUK2*PI4,0.0)*PHFAC
150   CONTINUE
160   CONTINUE
C
C     SUMMATION IN DIRECT SPACE
C     (MARADUDIN EQ.6.2.21)
C
      KDM1=2*KDM(1)+1
      KDM2=2*KDM(2)+1
      KDM3=2*KDM(3)+1
      DO 200 K1=1,KDM1
      FK1=FLOAT(K1-KDM(1)-1)
      DO 200 K2=1,KDM2
      FK2=FLOAT(K2-KDM(2)-1)
      DO 200 K3=1,KDM3
      FK3=FLOAT(K3-KDM(3)-1)
      DO 170 J=1,3
C     X(L')-X(L)
      XL(J)=FK1*A(J,1)+FK2*A(J,2)+FK3*A(J,3)
C     X(L'K')-X(LK)
      XLK(J)=XL(J)+XK(J)
170   CONTINUE
      Z=XLK(1)**2+XLK(2)**2+XLK(3)**2
C     SQRT(P)*/X(L'K')-X(LK)/
      ZR=SQRP*SQRT(Z)
      IF (ZR.GT.CONVRG) GO TO 200
C     IF X(L'K') = X(LK), AND XK =0, SKIP THIS TERM
      IF (RZERO.AND.ABS(FK1)+ABS(FK2)+ABS(FK3).LT.1.0E-10) GO TO 200
C
C     CALCULATE H(ALFA,BETA)
C
      EXQ=EXP(-ZR**2)
      A1=(1.0-erff(ZR)+2.0*ZR/PISQR*EXQ)/ZR**3*PVA
      A2=3.0*A1+4.0/PISQR*EXQ*PVA
      DO 180 I=1,3
      DO 180 J=1,3
      HAB(I,J)=XLK(I)*XLK(J)/Z*A2
C     EXTRA DIAGONAL TERM
      IF (I.EQ.J) HAB(I,J)=HAB(I,J)-A1
180   CONTINUE
C
      PHFAC=CEXP(CMPLX(.0,QPI(1)*XL(1)+QPI(2)*XL(2)+QPI(3)*XL(3)))
      DO 190 I=1,3
      DO 190 J=1,3
      BBB(I,J)=BBB(I,J)+PHFAC*CMPLX(-HAB(I,J),0.0)
190   CONTINUE
200   CONTINUE
C     FROM (6.2.53) WE GET AN EXTRA TERM FOR XK = 0
C     WHICH WAS EXCLUDED ABOVE
      IF (.NOT.RZERO) GO TO 220
      A1=-4.0/3.0/PISQR*PVA
      DO 210 I=1,3
      BBB(I,I)=BBB(I,I)+CMPLX(A1,0.0)
210   CONTINUE
C
C     PHASE FACTOR
220   PHFAC=CEXP(CMPLX(0.0,QPI(1)*XK(1)+QPI(2)*XK(2)+QPI(3)*XK(3)))
      DO 230 I=1,3
      DO 230 J=1,3
      BBB(I,J)=BBB(I,J)*PHFAC
230   CONTINUE
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE COUPL (A11,A12,A21,A22,A,I1,J1,K1,L1)
C-----------------------------------------------------------------------
      REAL A11(I1,J1), A12(I1,J1), A21(I1,J1), A22(I1,J1), A(K1,L1)
C
C     PUT THE MATRICES A11,A12,A21,A22 INTO THE BIG MATRIX A
C                 (A11  A12)
C             A = (A21  A22)
C
      DO 100 J=1,J1
      DO 100 I=1,I1
      K=I+I1
      L=J+J1
      A(I,J)=A11(I,J)
      A(I,L)=A12(I,J)
      A(K,J)=A21(I,J)
      A(K,L)=A22(I,J)
100   CONTINUE
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE CSSCAL (N, SA, CX, INCX)
C  Purpose:    Multiply a complex vector by a single-precision scalar,
C              y = ay.
C  Usage:      CALL CSSCAL (N, SA, CX, INCX)
C  Arguments:
C     N      - Length of vectors X.  (Input)
C     SA     - Real scalar.  (Input)
C     CX     - Complex vector of length MAX(N*IABS(INCX),1).
C                 (Input/Output)
C              CSSCAL replaces X(I) with SA*X(I) for I = 1,...,N.
C              X(I) refers to a specific element of CX.
C     INCX   - Displacement between elements of CX.  (Input)
C              X(I) is defined to be CX(1+(I-1)*INCX). INCX must be
C              greater than 0.
C-----------------------------------------------------------------------
C
      INTEGER    N, INCX
      REAL       SA
      COMPLEX    CX(*)
      INTEGER    I, NINCX
      INTRINSIC  CMPLX
      COMPLEX    CMPLX
C
      IF (N .GT. 0) THEN
         IF (INCX .NE. 1) THEN
C                                  CODE FOR INCREMENT NOT EQUAL TO 1
            NINCX = N*INCX
            DO 10  I=1, NINCX, INCX
               CX(I) = CMPLX(SA,0.0E0)*CX(I)
   10       CONTINUE
         ELSE
C                                  CODE FOR INCREMENT EQUAL TO 1
            DO 20  I=1, N
               CX(I) = CMPLX(SA,0.0E0)*CX(I)
   20       CONTINUE
         END IF
      END IF
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE MAT33 (A,B,C,D,E,F,G,H,P,Q,S)
C     PUT ELEMENTS INTO A 3X3 MATRIX, AND MULTIPLY BY PHASE FACTOR
C-----------------------------------------------------------------------
C
C             (A  B  C)
C      S = Q* (D  E  F)
C             (G  H  P)
C
      REAL S(3,3)
      S(1,1)=Q*A
      S(1,2)=Q*B
      S(1,3)=Q*C
      S(2,1)=Q*D
      S(2,2)=Q*E
      S(2,3)=Q*F
      S(3,1)=Q*G
      S(3,2)=Q*H
      S(3,3)=Q*P
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE MATOP (OPER,A,NA,MA,B,NB,MB,C,FACTOR)
C
C     MATRIX MANIPULATIONS
C
C     PARAMETERS:
C     OPER ... VARIABLE OF TYPE "INTEGER" CONTROLLING THE ACTION
C              TAKEN BY MATOP:
C              '.    ' - B=A*FACTOR
C              '+    ' - C=A+B
C              '-    ' - C=A-B
C              '*    ' - C=A*B
C              'INV  ' - B=INVERSE OF A (USING IMSL)
C              'TRANS' - B=(TRANSPOSE OF A)*FACTOR
C              'EIG  ' - HERMITIAN EIGENVALUE PROBLEM (USING IMSL)
C              THE COMPLEX MATRIX (REAL,IMAG) IS (A,B)
C              THE EIGENVALUES ARE STORED IN THE DIAGONAL OF C
C              THE EIGENVECTORS (REAL,IMAG) ARE STORED IN (A,B)
C     A ...... MATRIX OF DIMENSION (NA,MA)
C     B ...... MATRIX OF DIMENSION (NB,MB)
C     C ...... MATRIX OF DIMENSION (NA,MB)
C     FACTOR . REAL NUMBER USED AS A MULTIPLICATIVE FACTOR
C              IN SOME OF THE OPERATIONS
C-----------------------------------------------------------------------
      CHARACTER OPER*5
      REAL A(NA,MA),B(NB,MB),C(NA,MB)

      REAL LAMBDA(12)
c     COMPLEX V(6,6),AA(6,6)
      real v(12,12),aa(12,12)

C DETERMINE OPERATION TYPE
      IF (OPER.EQ.'.    ') GO TO 110
      IF (OPER.EQ.'+    '.OR.OPER.EQ.'-    ') GO TO 150
      IF (OPER.EQ.'*    ') GO TO 180
      IF (OPER.EQ.'INV  ') GO TO 210
      IF (OPER.EQ.'TRANS') GO TO 270
      IF (OPER.EQ.'EIG  ') GO TO 300
      WRITE (6,100) OPER ! operation error
100   FORMAT (37H0IMPROPER OPERATION IN MATOP, OPER = ,A5)
      STOP

C *******  SCALAR MULTIPLICATION B = A*FACTOR  *************
C B MAY EQUAL A IN THIS CASE
C C IS A DUMMY ARRAY IN THIS CASE

110   IF (NA.EQ.NB.AND.MA.EQ.MB) GO TO 130
      WRITE (6,120) NA,MA,NB,MB ! dimension error
120   FORMAT (40H0DIMENSION ERROR IN MATOP, NA,MA,NB,MB =,4I10)
      STOP
130   DO 140 J=1,MA
      DO 140 I=1,NA
      B(I,J)=A(I,J)*FACTOR
140   CONTINUE
      RETURN

C ******* MATRIX ADDITION *******  
C C MAY EQUAL A OR B IN CALLING MATOP
C FACTOR IS A DUMMY

150   SIGN=1.0
      IF (OPER.EQ.'-    ') SIGN=-1.0
      IF (NA.EQ.NB.AND.MA.EQ.MB) GO TO 160
      WRITE (6,120) NA,MA,NB,MB  ! dimension error
      STOP
160   DO 170 J=1,MA
      DO 170 I=1,NA
      C(I,J)=A(I,J)+B(I,J)*SIGN
170   CONTINUE
      RETURN
C
C ******* MATRIX MULTIPLICATION C = A*B  ******* 
C C MUST NOT EQUAL A OR B IN CALLING MATOP
C FACTOR IS A DUMMY

180   IF (MA.EQ.NB) GO TO 190
      WRITE (6,120) NA,MA,NB,MB ! dimension error
      STOP
190   DO 200 J=1,MB
      DO 200 I=1,NA
      C(I,J)=0.0
      DO 200 K=1,MA
      C(I,J)=C(I,J)+A(I,K)*B(K,J)
200   CONTINUE
      RETURN

C ******* MATRIX INVERSION B = A**(-1)   USING LINRG (IMSL)  ******* 
C B MAY EQUAL A IN CALLING MATOP
C C AND FACTOR ARE DUMMIES

210   IF (NA.NE.MA.OR.MA.NE.NB.OR.NB.NE.MB) THEN
        WRITE (6,120) NA,MA,NB,MB    ! dimension error
        STOP
      ENDIF
220   IF (NA.GT.12) THEN
        WRITE (6,230)
230     FORMAT (49H0NA.GT.12 IN MATOP, PLEASE INCREASE DIMENSION L,M)
        STOP
      ELSE
c        CALL LINRG(NA,A,NA,B,NA)
        call invmat(a,b,na,na)
      ENDIF
      RETURN

C ******* TRANSPOSE B = A(T)*FACTOR  ******* 
C B MAY NOT EQUAL A
C C IS A DUMMY

270   IF (NA.EQ.MB.AND.MA.EQ.NB) GO TO 280
      WRITE (6,120) NA,MA,NB,MB ! dimension error
      STOP
280   DO 290 J=1,MA
      DO 290 I=1,NA
      B(J,I)=A(I,J)*FACTOR
290   CONTINUE
      RETURN

C ******* HERMITIAN EIGENVALUE PROBLEM  ******* 
300   IF (NA.EQ.MA.AND.MA.EQ.NB.AND.NB.EQ.MB) GO TO 310
      WRITE (6,120) NA,MA,NB,MB  ! dimension error
      STOP
310   IF (NA.GT.6) THEN
        WRITE (6,320)
320     FORMAT (43H0PLEASE INCREASE DIM LAMBDA,VR,VI IN MATOP/)
        STOP
      ELSE
        DO J=1,na
          DO I=1,na
            AA(I,J)=A(I,J)
            aa(i,j+na) = -b(i,j)
            aa(i+na,j) = b(i,j)
            aa(i+na,j+na) = a(i,j)
          END DO
        END DO
        na2 = 2*na
	
        call eigen(AA,na2,na2,LAMBDA,V)

c ***** eigenvectors should already be normalized
c        DO J=1,6
c          WK1=1.0/SCNRM2(6,V(1,J),1)  ! normalize eigenvectors
c          CALL CSSCAL(6,WK1,V(1,J),1)
c        END DO
        DO J=1,na
          C(J,J)=LAMBDA(2*J-1)
          DO I=1,NA
            A(I,J)=V(i,2*j-1)
            B(I,J)=V(i+na,2*j-1)
          END DO
        END DO
      ENDIF
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE PARBCM (ITYPE,IPRINT)
C
C     INITIALIZE THE MODEL PARAMETERS
C
C     INPUT:
C     ITYPE ... 1-DIAMOND (default parameters)
C               2-SILICON  (default parameters)
C               3-GERMANIUM (natural, M=72.59, default parameters)
C               4-ALFA-TIN  (default parameters)
C               5-Germanium 70 (default parameters)
C               6-GE76  (default parameters)
C               11- Diamond with pars set in NEWPAR
C               12,13,14,15, etc
C     IPRINT .. 0-PARAMETERS NOT PRINTED
C               ELSE- PARAMETERS PRINTED ON TAPE6.
C
C     OUTPUT: 1/3 OF A PAGE ON TR12APE6, IF SPECIFIED BY IPRINT.
C-----------------------------------------------------------------------

      COMMON /PRAM/ FCBC(6),FCION(6),MASS,SCALE,ACELL2,Z2EPS,EXPRAM,THR
     1     AM,CELAST(3,2)
      REAL MASS


C     A ... PRIMITIVE TRANSLATIONAL VECTORS.
C           A(I,J) IS COORDINATE I OF THE J'TH VECTOR.
C     R12 . CONTAINS ALL THE RELATIVE COORDINATES OF THE IONS AND
C           BOND CHARGES, I.E. X(K')-X(K).
C           R12(I,J) IS COORDINATE J OF RELATIVE VECTOR I.

      COMMON /BASIS/ A(3,3),R12(12,3)
      data a/1.,1.,0.,1.,0.,1.,0.,1.,1./ !1st 3 are 1st column
      data r12/ 0., .5, .25,-.25, .25,-.25, .5, .0, .5,-.5, .0, .5,
     1          0., .5, .25, .25,-.25,-.25, .0, .5, .5, .5, .5, .0,
     2          0., .5, .25,-.25,-.25, .25, .5, .5, .0, .0,-.5,-.5/

      DATA EESU/4.80325/,AMU24/1.66053/
      DATA PI/3.1415926535898/

      ITIPE=MOD(ITYPE-1,10)+1     ! next 100 lines changed by STZO

      IF (ITIPE.EQ.1) THEN        ! DIAMOND PARAMETERS
        IF (ITYPE.EQ.1) THEN
          FCBC(1)=51.682
          FCBC(2)=45.256
          FCBC(3)=4.669
          FCBC(4)=3.138
          FCBC(5)=-1.607
          FCBC(6)=3.138
          FCION(1)=-18.161
          FCION(2)=-6.02
          FCION(3)=0.0
          FCION(4)=0.0
          FCION(5)=0.0
          FCION(6)=0.0
          Z2EPS=0.885
        ENDIF
        MASS=12.0
        ACELL2=1.78
C     WARREN ET.AL., INELASTIC SCATTERING OF NEUTRONS, CONF. BOMBAY 1964
C                    IAEA VIENNA 1965, VOL.1,P.361
C    ALSO: BESERMAN (1972) (RAMAN, UNPUBLISHED)
        EXPRAM=39.96
C     MAC SKIMIN, PHYS.REV.105,116 (1957)
        CELAST(1,1)=10.76
        CELAST(2,1)=1.25
        CELAST(3,1)=5.76
      ELSE IF (ITIPE.EQ.2) THEN  !  SILICON PARAMETERS
        IF (ITYPE.EQ.2) THEN
          FCBC(1)=10.77
          FCBC(2)=2.17
          FCBC(3)=2.15
          FCBC(4)=2.15
          FCBC(5)=-2.15
          FCBC(6)=2.15
          FCION(1)=4.57
          FCION(2)=7.03
          FCION(3)=0.0
          FCION(4)=0.0
          FCION(5)=0.0
          FCION(6)=0.0
          Z2EPS=0.179
        ENDIF
        MASS=28.086
        ACELL2=2.715
C     DOLLING, INELASTIC SCATTERING OF NEUTRONS IN SOLIDS AND LIQUIDS,
C              IAEA VIENNA 1963, VOL.II, P.37
C              IAEA VIENNA 1965, VOL.I , P.249
        EXPRAM=15.53
C     MAC SKIMIN, PHYS.REV.105, 116 (1957)
C                 J.APPL.PHYS.24, 988 (1953)
        CELAST(1,1)=1.6563
        CELAST(2,1)=0.6390
        CELAST(3,1)=0.7956
      ELSE IF (ITIPE.EQ.3) THEN  !  GERMANIUM PARAMETERS
        IF (ITYPE.EQ.3) THEN
          FCBC(1)=9.9075
          FCBC(2)=1.5075
          FCBC(3)=2.10
          FCBC(4)=2.10
          FCBC(5)=-2.10
          FCBC(6)=2.10
          FCION(1)=5.1354
          FCION(2)=7.3512
          FCION(3)=0.0
          FCION(4)=0.0
          FCION(5)=0.0
          FCION(6)=0.0
          Z2EPS=0.162
        ENDIF
        MASS=72.59
        ACELL2=2.825
C     NILSSON, PHYS.REV.B3, 364 (1971)
        EXPRAM=9.11
C     MAC SKIMIN, J.APPL.PHYS.24, 988 (1953)
        CELAST(1,1)=1.2897
        CELAST(2,1)=0.48346
        CELAST(3,1)=0.67130
      ELSE IF (ITIPE.EQ.4) THEN  !  ALFA-TIN PARAMETERS
        IF (ITYPE.EQ.4) THEN
          FCBC(1)=9.49
          FCBC(2)=1.69
          FCBC(3)=1.95
          FCBC(4)=1.95
          FCBC(5)=-1.95
          FCBC(6)=1.95
          FCION(1)=5.94
          FCION(2)=8.18
          FCION(3)=0.0
          FCION(4)=0.0
          FCION(5)=0.0
          FCION(6)=0.0
          Z2EPS=0.163
        ENDIF
        MASS=118.69
        ACELL2=3.23
C     BUCHENAUER, PHYS.REV.B3, 1243 (1971)
        EXPRAM=5.97
C     PRICE, PHYS.REV.B1, 1268 (1971)
        CELAST(1,1)=0.690
        CELAST(2,1)=0.293
        CELAST(3,1)=0.362
      ELSE IF (ITIPE.EQ.5) THEN  !  GERMANIUM 70 PARAMETERS
        IF (ITYPE.EQ.5) THEN     !  added by STZO
          FCBC(1)=9.9075   ! These are just like natural Ge
          FCBC(2)=1.5075   !
          FCBC(3)=2.10     !
          FCBC(4)=2.10     !
          FCBC(5)=-2.10    !
          FCBC(6)=2.10     !
          FCION(1)=5.1354  !
          FCION(2)=7.3512  !
          FCION(3)=0.0     !
          FCION(4)=0.0     !
          FCION(5)=0.0     !
          FCION(6)=0.0     !
          Z2EPS=0.162      ! These are just like natural Ge
        ENDIF
        MASS=70.00   ! only the mass is different
        ACELL2=2.825
C     NILSSON, PHYS.REV.B3, 364 (1971)
        EXPRAM=9.11  ! natural Ge
C     MAC SKIMIN, J.APPL.PHYS.24, 988 (1953)
        CELAST(1,1)=1.2897      ! natural Ge
        CELAST(2,1)=0.48346     ! natural Ge
        CELAST(3,1)=0.67130     ! natural Ge
      ELSE IF (ITIPE.EQ.6) THEN  !  GERMANIUM 76 PARAMETERS
        IF (ITYPE.EQ.6) THEN     !  added by STZO
          FCBC(1)=9.9075   ! These are just like natural Ge
          FCBC(2)=1.5075   !
          FCBC(3)=2.10     !
          FCBC(4)=2.10     !
          FCBC(5)=-2.10    !
          FCBC(6)=2.10     !
          FCION(1)=5.1354  !
          FCION(2)=7.3512  !
          FCION(3)=0.0     !
          FCION(4)=0.0     !
          FCION(5)=0.0     !
          FCION(6)=0.0     !
          Z2EPS=0.162      ! These are just like natural Ge
        ENDIF
        MASS=75.69   ! only the mass is different
        ACELL2=2.825
C     NILSSON, PHYS.REV.B3, 364 (1971)
        EXPRAM=9.11  ! natural Ge
C     MAC SKIMIN, J.APPL.PHYS.24, 988 (1953)
        CELAST(1,1)=1.2897      ! natural Ge
        CELAST(2,1)=0.48346     ! natural Ge
        CELAST(3,1)=0.67130     ! natural Ge
      ENDIF
C
C     CALCULATE DERIVED QUANTITIES
C
      FAK=4.0*ACELL2**4/EESU**2
C     CONVERT INTO UNITS E**2/VCELL
      DO 190 I=1,3
      CELAST(I,1)=CELAST(I,1)*FAK
190   CONTINUE
C     SCALEFACTOR FOR FREQUENCY NU IN THZ (NOT OMEGA=2*PI*NU ||)
      SCALE=1.0E4/(2.0*PI)**2*EESU**2/((2.0*ACELL2)**3/4.0)/AMU24
C     THEORETICAL ELASTIC CONSTANTS
C     Z2EPS = Z**2 / EPSILON
      P1=0.5*FCBC(1)+FCION(1)
      CELAST(1,2)=P1+2.*FCBC(3)+2.218*Z2EPS+8.*FCION(3)
      CELAST(2,2)=-P1+FCBC(2)+2.*FCION(2)+2.*FCBC(4)-FCBC(3)-FCBC(5)-30.
     1765*Z2EPS+8.0*FCION(4)-4.0*FCION(3)-4.0*FCION(5)
      CELAST(3,2)=P1-(FCBC(2)/2.+FCION(2)-17.653*Z2EPS)**2/P1+FCBC(3)+FC
     1BC(5)-1.109*Z2EPS+4.*FCION(3)+4.*FCION(5)
C     THEORETICAL RAMAN FREQUENCY
      THRAM=SQRT((4.0*FCBC(1)+8.0*FCION(1))*SCALE/MASS)
      IF (IPRINT.EQ.0) RETURN
C
C     PRINT OUT PARAMETERS
C
c     WRITE (6,200) ITIPE,FCBC,FCION,MASS,Z2EPS,ACELL2,
c    &              EXPRAM,THRAM,CELAST
200   FORMAT (1H ,'PARAMETERS FOR MATERIAL NO. ',I3,/
     &        1H ,'BOND CHARGE FORCES = ',6F12.4,' E**2/VCELL',/,
     &        1H ,'ION FORCES = ',8X,6F12.4,' E**2/VCELL',/,
     &        1H ,'MASS = ',F12.4,' AMU',/,
     &        1H ,'Z**2/EPSILON= ',F12.4,/,
     &        1H ,'1/2 CUBE LENGTH = ',F12.4,' ANGSTROM',/
     &        1H ,'EXP.RAMAN FREQUENCY = ',F12.4,' THZ',/,
     &        1H ,'TH. RAMAN FREQUENCY = ',F12.4,' THZ,'/
     &        1H ,'EXP.ELASTIC CONSTANTS C11,C12,C44 = ',3F12.4,
     &                       'E**2/VCELL',/,
     &        1H ,'TH. ELASTIC CONSTANTS C11,C12,C44 = ',3F12.4,
     &                       'E**2/VCELL')
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE RMAT (RR,RI)
C
C     ION-ION FORCE DYNAMICAL MATRIX, DENOTED "R" IN W.WEBER'S ARTICLE
C     OUTPUT:
C     RR,RI ... REAL & IMAG PART OF MATRIX R
C-----------------------------------------------------------------------
      REAL RR(6,6), RI(6,6)

      COMMON /PRAM/ FCBC(6),FCION(6),MASS,SCALE,ACELL2,Z2EPS,EXPRAM,THR
     1AM,CELAST(3,2)
      REAL MASS
      COMMON /QVECT/ QPI(3)
      REAL MUP4,NUP4,LAMBP4
      COSX2=COS(QPI(1)/2.0)
      SINX2=SIN(QPI(1)/2.0)
      COSY2=COS(QPI(2)/2.0)
      SINY2=SIN(QPI(2)/2.0)
      COSZ2=COS(QPI(3)/2.0)
      SINZ2=SIN(QPI(3)/2.0)
      DO 100 J=1,6
      DO 100 I=1,6
      RR(I,J)=0.0
      RI(I,J)=0.0
100   CONTINUE
      ALFA4=4.0*FCBC(1)
      ALFAP4=4.0*FCION(1)
      BETAP4=4.0*FCION(2)
      MUP4=4.0*FCION(3)
      NUP4=4.0*FCION(4)
      LAMBP4=4.0*FCION(5)
      DELTP4=4.0*FCION(6)
C
C     CONSTRUCT MATRIX R FOR 1.N.N.
C
C     BY TRANSLATIONAL INVARIANCE WE HAVE:
      DO 110 I=1,6
      RR(I,I)=ALFAP4+ALFA4
110   CONTINUE
      RR(1,4)=-ALFAP4*COSX2*COSY2*COSZ2
      RR(2,5)=RR(1,4)
      RR(3,6)=RR(1,4)
      RR(1,5)=BETAP4*SINX2*SINY2*COSZ2
      RR(1,6)=BETAP4*SINX2*COSY2*SINZ2
      RR(2,6)=BETAP4*COSX2*SINY2*SINZ2
      RR(2,4)=RR(1,5)
      RR(3,4)=RR(1,6)
      RR(3,5)=RR(2,6)
      RI(1,4)=ALFAP4*SINX2*SINY2*SINZ2
      RI(2,5)=RI(1,4)
      RI(3,6)=RI(1,4)
      RI(1,5)=-BETAP4*COSX2*COSY2*SINZ2
      RI(1,6)=-BETAP4*COSX2*SINY2*COSZ2
      RI(2,6)=-BETAP4*SINX2*COSY2*COSZ2
      RI(2,4)=RI(1,5)
      RI(3,4)=RI(1,6)
      RI(3,5)=RI(2,6)
C
C     CONSTRUCT MATRIX R FOR 2.N.N.
C
      COSX=COS(QPI(1))
      SINX=SIN(QPI(1))
      COSY=COS(QPI(2))
      SINY=SIN(QPI(2))
      COSZ=COS(QPI(3))
      SINZ=SIN(QPI(3))
      RR2=MUP4*(2.0-COSX*(COSY+COSZ))+LAMBP4*(1.0-COSY*COSZ)
      RR(1,1)=RR(1,1)+RR2
      RR(4,4)=RR(4,4)+RR2
      RR2=MUP4*(2.0-COSY*(COSZ+COSX))+LAMBP4*(1.0-COSZ*COSX)
      RR(2,2)=RR(2,2)+RR2
      RR(5,5)=RR(5,5)+RR2
      RR2=MUP4*(2.0-COSZ*(COSX+COSY))+LAMBP4*(1.0-COSX*COSY)
      RR(3,3)=RR(3,3)+RR2
      RR(6,6)=RR(6,6)+RR2
      RR2=NUP4*SINX*SINY
      RR(1,2)=RR(1,2)+RR2
      RR(4,5)=RR(4,5)+RR2
      RR2=NUP4*SINX*SINZ
      RR(1,3)=RR(1,3)+RR2
      RR(4,6)=RR(4,6)+RR2
      RR2=NUP4*SINY*SINZ
      RR(2,3)=RR(2,3)+RR2
      RR(5,6)=RR(5,6)+RR2
      RI(1,2)=-DELTP4*SINZ*(COSX-COSY)
      RI(1,3)=-DELTP4*SINY*(COSX-COSZ)
      RI(2,3)=-DELTP4*SINX*(COSY-COSZ)
      RI(4,5)=-RI(1,2)
      RI(4,6)=-RI(1,3)
      RI(5,6)=-RI(2,3)
      DO 120 I=1,5
      I1=I+1
      DO 120 J=I1,6
      RI(J,I)=-RI(I,J)
      RR(J,I)=RR(I,J)
120   CONTINUE
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE SMATX (S)
C
C     BC-BC FORCE DYNAMICAL MATRIX, DENOTED "S" IN W.WEBER'S ARTICLE
C     OUTPUT:
C     S ... THE MATRIX S (A REAL MATRIX)
C-----------------------------------------------------------------------
      REAL S(12,12)

      COMMON /PRAM/ FCBC(6),FCION(6),MASS,SCALE,ACELL2,Z2EPS,EXPRAM,THR
     1AM,CELAST(3,2)
      REAL MASS
      COMMON /QVECT/ QPI(3)
      REAL A11(3,3), A12(3,3), A21(3,3), A22(3,3), B11(6,6), B12(6,
     16), B21(6,6), B22(6,6)
      A=2.0*FCBC(1)+4.0*FCBC(3)+2.0*FCBC(5)
      B=2.0*FCBC(2)+2.0*FCBC(4)
      C=FCBC(3)
      D=FCBC(4)
      E=FCBC(5)
      F=FCBC(6)
      CALL MAT33 (A,B,B,B,A,B,B,B,A,1.0,A11)
      CALL MAT33 (A,-B,B,-B,A,-B,B,-B,A,1.0,A22)
      Q=-2.0*COS((QPI(1)+QPI(3))/2.0)
      CALL MAT33 (C,F,D,-F,E,-F,D,F,C,Q,A12)
      DO 100 I=1,3
      DO 100 J=1,3
      A21(J,I)=A12(I,J)
100   CONTINUE
      CALL COUPL (A11,A12,A21,A22,B11,3,3,6,6)
      CALL MAT33 (A,-B,-B,-B,A,B,-B,B,A,1.,A11)
      CALL MAT33 (A,B,-B,B,A,-B,-B,-B,A,1.,A22)
      Q=-2.0*COS((QPI(1)-QPI(3))/2.0)
      CALL MAT33 (C,-F,-D,F,E,-F,-D,F,C,Q,A12)
      DO 110 I=1,3
      DO 110 J=1,3
      A21(J,I)=A12(I,J)
110   CONTINUE
      CALL COUPL (A11,A12,A21,A22,B22,3,3,6,6)
      Q=-2.0*COS((QPI(2)+QPI(3))/2.0)
      CALL MAT33 (E,-F,-F,F,C,D,F,D,C,Q,A11)
      Q=-2.0*COS((QPI(1)+QPI(2))/2.0)
      CALL MAT33 (C,D,F,D,C,F,-F,-F,E,Q,A12)
      Q=-2.0*COS((QPI(1)-QPI(2))/2.0)
      CALL MAT33 (C,-D,F,-D,C,-F,-F,F,E,Q,A21)
      Q=-2.0*COS((QPI(2)-QPI(3))/2.0)
      CALL MAT33 (E,F,-F,-F,C,-D,F,-D,C,Q,A22)
      CALL COUPL (A11,A12,A21,A22,B12,3,3,6,6)
      DO 120 I=1,6
      DO 120 J=1,6
      B21(J,I)=B12(I,J)
120   CONTINUE
      CALL COUPL (B11,B12,B21,B22,S,6,6,12,12)
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE TMAT (TR,TI)
      REAL TR(6,12),TI(6,12)
C
C     ION-BC FORCE DYNAMICAL MATRIX, DENOTED "T" IN W.WEBER'S ARTICLE
C     OUTPUT:
C     TR,TI ... REAL & IMAG PART OF MATRIX T
C-----------------------------------------------------------------------

      COMMON /PRAM/ FCBC(6),FCION(6),MASS,SCALE,ACELL2,Z2EPS,EXPRAM,THR
     1AM,CELAST(3,2)
      REAL MASS
      COMMON /QVECT/ QPI(3)
      REAL A11(3,3), A12(3,3), A21(3,3), A22(3,3), TR1(6,6), TR2(6,
     16)
      A=FCBC(1)
      B=FCBC(2)
      Q=-COS((QPI(1)+QPI(2)+QPI(3))/4.0)
      CALL MAT33 (A,B,B,B,A,B,B,B,A,Q,A11)
      Q=-COS((QPI(1)-QPI(2)+QPI(3))/4.0)
      CALL MAT33 (A,-B,B,-B,A,-B,B,-B,A,Q,A12)
      CALL COUPL (A11,A12,A11,A12,TR1,3,3,6,6)
      Q=-COS((QPI(1)-QPI(2)-QPI(3))/4.0)
      CALL MAT33 (A,-B,-B,-B,A,B,-B,B,A,Q,A11)
      Q=-COS((QPI(1)+QPI(2)-QPI(3))/4.0)
      CALL MAT33 (A,B,-B,B,A,-B,-B,-B,A,Q,A12)
      CALL COUPL (A11,A12,A11,A12,TR2,3,3,6,6)
      DO 100 J=1,6
      DO 100 I=1,6
      TR(I,J)=TR1(I,J)
      TR(I,J+6)=TR2(I,J)
100   CONTINUE
      Q=-SIN((QPI(1)+QPI(2)+QPI(3))/4.0)
      CALL MAT33 (A,B,B,B,A,B,B,B,A,Q,A11)
      CALL MAT33 (A,B,B,B,A,B,B,B,A,-Q,A21)
      Q=SIN((QPI(1)-QPI(2)+QPI(3))/4.0)
      CALL MAT33 (A,-B,B,-B,A,-B,B,-B,A,Q,A12)
      CALL MAT33 (A,-B,B,-B,A,-B,B,-B,A,-Q,A22)
      CALL COUPL (A11,A12,A21,A22,TR1,3,3,6,6)
      Q=-SIN((QPI(1)-QPI(2)-QPI(3))/4.0)
      CALL MAT33 (A,-B,-B,-B,A,B,-B,B,A,Q,A11)
      CALL MAT33 (A,-B,-B,-B,A,B,-B,B,A,-Q,A21)
      Q=SIN((QPI(1)+QPI(2)-QPI(3))/4.0)
      CALL MAT33 (A,B,-B,B,A,-B,-B,-B,A,Q,A12)
      CALL MAT33 (A,B,-B,B,A,-B,-B,-B,A,-Q,A22)
      CALL COUPL (A11,A12,A21,A22,TR2,3,3,6,6)
      DO 110 J=1,6
      DO 110 I=1,6
      TI(I,J)=TR1(I,J)
      TI(I,J+6)=TR2(I,J)
110   CONTINUE
      RETURN
      END


C-----------------------------------------------------------------------
      subroutine eigen(a,n,np,eival,eivect)
C-----------------------------------------------------------------------

      real a(np,np),eival(np),eivect(np,np)
      real d(12),e(12)
c
      do j=1,np
        do i=1,np
          eivect(i,j) = a(i,j)
        enddo
      enddo
c
      call tred2(eivect,n,np,d,e)
      call tqli(d,e,n,np,eivect)
      call eigsrt(d,eivect,n,np)
c
      do i=1,np
        eival(i) = d(i)
      enddo
c
      return
      end


C-----------------------------------------------------------------------
      SUBROUTINE TRED2(A,N,NP,D,E)
C-----------------------------------------------------------------------

      real A(NP,NP),D(NP),E(NP)
c
      IF(N.GT.1)THEN
        DO 18 I=N,2,-1
          L=I-1
          H=0.
          SCALE=0.
          IF(L.GT.1)THEN
            DO 11 K=1,L
              SCALE=SCALE+ABS(A(I,K))
11          CONTINUE
            IF(SCALE.EQ.0.)THEN
              E(I)=A(I,L)
            ELSE
              DO 12 K=1,L
                A(I,K)=A(I,K)/SCALE
                H=H+A(I,K)**2
12            CONTINUE
              F=A(I,L)
              G=-SIGN(SQRT(H),F)
              E(I)=SCALE*G
              H=H-F*G
              A(I,L)=F-G
              F=0.
              DO 15 J=1,L
                A(J,I)=A(I,J)/H
                G=0.
                DO 13 K=1,J
                  G=G+A(J,K)*A(I,K)
13              CONTINUE
                IF(L.GT.J)THEN
                  DO 14 K=J+1,L
                    G=G+A(K,J)*A(I,K)
14                CONTINUE
                ENDIF
                E(J)=G/H
                F=F+E(J)*A(I,J)
15            CONTINUE
              HH=F/(H+H)
              DO 17 J=1,L
                F=A(I,J)
                G=E(J)-HH*F
                E(J)=G
                DO 16 K=1,J
                  A(J,K)=A(J,K)-F*E(K)-G*A(I,K)
16              CONTINUE
17            CONTINUE
            ENDIF
          ELSE
            E(I)=A(I,L)
          ENDIF
          D(I)=H
18      CONTINUE
      ENDIF
      D(1)=0.
      E(1)=0.
      DO 23 I=1,N
        L=I-1
        IF(D(I).NE.0.)THEN
          DO 21 J=1,L
            G=0.
            DO 19 K=1,L
              G=G+A(I,K)*A(K,J)
19          CONTINUE
            DO 20 K=1,L
              A(K,J)=A(K,J)-G*A(K,I)
20          CONTINUE
21        CONTINUE
        ENDIF
        D(I)=A(I,I)
        A(I,I)=1.
        IF(L.GE.1)THEN
          DO 22 J=1,L
            A(I,J)=0.
            A(J,I)=0.
22        CONTINUE
        ENDIF
23    CONTINUE
      RETURN
      END


C-----------------------------------------------------------------------
      SUBROUTINE TQLI(D,E,N,NP,Z)
C-----------------------------------------------------------------------

      real D(NP),E(NP),Z(NP,NP)
c
      IF (N.GT.1) THEN

        DO 11 I=2,N

          E(I-1)=E(I)
11      CONTINUE
        E(N)=0.
        DO 15 L=1,N
          ITER=0
1         DO 12 M=L,N-1
            DD=ABS(D(M))+ABS(D(M+1))
            IF (ABS(E(M))+DD.EQ.DD) GO TO 2
12        CONTINUE
          M=N
2         IF(M.NE.L)THEN
	     
            IF(MOD(ITER+1,100).EQ.0) THEN	       	       
c	       write(*,*) (E(I),I=L,M)
c	       write(*,*) (D(I),I=L,M)
	       write(*,*)  'TQLI warning: too many iterations '	       	       
	    ENDIF   
            ITER=ITER+1
            G=(D(L+1)-D(L))/(2.*E(L))
            R=SQRT(G**2+1.)
            G=D(M)-D(L)+E(L)/(G+SIGN(R,G))
            S=1.
            C=1.
            P=0.
            DO 14 I=M-1,L,-1
              F=S*E(I)
              B=C*E(I)
              if(abs(f).ge.abs(g))then
                c=g/f
                r=sqrt(c**2+1.)
                e(i+1)=f*r
                s=1./r
                c=c*s
              ELSE
                S=F/G
                R=SQRT(S**2+1.)
                E(I+1)=G*R
                C=1./R
                S=S*C
              ENDIF
              G=D(I+1)-P
              R=(D(I)-G)*S+2.*C*B
              P=S*R
              D(I+1)=G+P
              G=C*R-B
              DO 13 K=1,N
                F=Z(K,I+1)
                Z(K,I+1)=S*Z(K,I)+C*F
                Z(K,I)=C*Z(K,I)-S*F
13            CONTINUE
14          CONTINUE
            D(L)=D(L)-P
            E(L)=G
            E(M)=0.
            GO TO 1
          ENDIF
15      CONTINUE
      ENDIF
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE EIGSRT(D,V,N,NP)
C-----------------------------------------------------------------------
c
      DIMENSION D(NP),V(NP,NP)
c
      DO 13 I=1,N-1
        K=I
        P=D(I)
        DO 11 J=I+1,N
          IF(D(J).GE.P)THEN
            K=J
            P=D(J)
          ENDIF
11      CONTINUE
        IF(K.NE.I)THEN
          D(K)=D(I)
          D(I)=P
          DO 12 J=1,N
            P=V(J,I)
            V(J,I)=V(J,K)
            V(J,K)=P
12        CONTINUE
        ENDIF
13    CONTINUE
      RETURN
      END


C-----------------------------------------------------------------------
      REAL*4 FUNCTION MYERF(X)
C renamed by J.S. => avoid conflict with some intrinsic libraries 
C-----------------------------------------------------------------------
      IF(X.LT.0.)THEN
        MYERF=-GAMMP(.5,X**2)
      ELSE
        MYERF=GAMMP(.5,X**2)
      ENDIF
      RETURN
      END
c
C-----------------------------------------------------------------------
      FUNCTION GAMMP(A,X)
C-----------------------------------------------------------------------
      IF(X.LT.0..OR.A.LE.0.)PAUSE
      IF(X.LT.A+1.)THEN
        CALL GSER(GAMSER,A,X,GLN)
        GAMMP=GAMSER
      ELSE
        CALL GCF(GAMMCF,A,X,GLN)
        GAMMP=1.-GAMMCF
      ENDIF
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE GCF(GAMMCF,A,X,GLN)
C-----------------------------------------------------------------------
      PARAMETER (ITMAX=100,EPS=3.E-7)
c
      GLN=GAMMLN(A)
      GOLD=0.
      A0=1.
      A1=X
      B0=0.
      B1=1.
      FAC=1.
      DO 11 N=1,ITMAX
        AN=FLOAT(N)
        ANA=AN-A
        A0=(A1+A0*ANA)*FAC
        B0=(B1+B0*ANA)*FAC
        ANF=AN*FAC
        A1=X*A0+ANF*A1
        B1=X*B0+ANF*B1
        IF(A1.NE.0.)THEN
          FAC=1./A1
          G=B1*FAC
          IF(ABS((G-GOLD)/G).LT.EPS)GO TO 1
          GOLD=G
        ENDIF
11    CONTINUE
      PAUSE 'A too large, ITMAX too small'
1     GAMMCF=EXP(-X+A*ALOG(X)-GLN)*G
      RETURN
      END

C-----------------------------------------------------------------------
      FUNCTION GAMMLN(XX)
C-----------------------------------------------------------------------
      REAL*8 COF(6),STP,HALF,ONE,FPF,X,TMP,SER
      DATA COF,STP/76.18009173D0,-86.50532033D0,24.01409822D0,
     *    -1.231739516D0,.120858003D-2,-.536382D-5,2.50662827465D0/
      DATA HALF,ONE,FPF/0.5D0,1.0D0,5.5D0/
c
      X=XX-ONE
      TMP=X+FPF
      TMP=(X+HALF)*LOG(TMP)-TMP
      SER=ONE
      DO 11 J=1,6
        X=X+ONE
        SER=SER+COF(J)/X
11    CONTINUE
      GAMMLN=TMP+LOG(STP*SER)
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE GSER(GAMSER,A,X,GLN)
C-----------------------------------------------------------------------
      PARAMETER (ITMAX=100,EPS=3.E-7)
c
      GLN=GAMMLN(A)
      IF(X.LE.0.)THEN
        IF(X.LT.0.)PAUSE
        GAMSER=0.
        RETURN
      ENDIF
      AP=A
      SUM=1./A
      DEL=SUM
      DO 11 N=1,ITMAX
        AP=AP+1.
        DEL=DEL*X/AP
        SUM=SUM+DEL
        IF(ABS(DEL).LT.ABS(SUM)*EPS)GO TO 1
11    CONTINUE
      PAUSE 'A too large, ITMAX too small'
1     GAMSER=SUM*EXP(-X+A*LOG(X)-GLN)
      RETURN
      END


C-----------------------------------------------------------------------
      subroutine invmat(a,b,n,np)
C-----------------------------------------------------------------------
      real a(np,np),b(np,np)
      real c(20,20)
      integer indx(12)
c
      do j=1,n
        do i=1,n
          b(i,j) = .0
          c(i,j) = a(i,j)
        enddo
        b(j,j) = 1.
      enddo
c
      call ludcmp(a,n,np,indx,d)
c
      do j=1,np
        call lubksb(a,n,np,indx,b(1,j))
      enddo
c
      do j=1,n
        do i=1,n
          a(i,j) = c(i,j)
        enddo
      enddo
      return
      end

C-----------------------------------------------------------------------
       SUBROUTINE LUDCMP(A,N,NP,INDX,D)
C-----------------------------------------------------------------------
      PARAMETER (NMAX=100,TINY=1.0E-20)
      DIMENSION A(NP,NP),INDX(N),VV(NMAX)
c
      D=1.
      DO 12 I=1,N
        AAMAX=0.
        DO 11 J=1,N
          IF (ABS(A(I,J)).GT.AAMAX) AAMAX=ABS(A(I,J))
11      CONTINUE
        IF (AAMAX.EQ.0.) PAUSE 'Singular matrix.'
        VV(I)=1./AAMAX
12    CONTINUE
      DO 19 J=1,N
        IF (J.GT.1) THEN
          DO 14 I=1,J-1
            SUM=A(I,J)
            IF (I.GT.1)THEN
              DO 13 K=1,I-1
                SUM=SUM-A(I,K)*A(K,J)
13            CONTINUE
              A(I,J)=SUM
            ENDIF
14        CONTINUE
        ENDIF
        AAMAX=0.
        DO 16 I=J,N
          SUM=A(I,J)
          IF (J.GT.1)THEN
            DO 15 K=1,J-1
              SUM=SUM-A(I,K)*A(K,J)
15          CONTINUE
            A(I,J)=SUM
          ENDIF
          DUM=VV(I)*ABS(SUM)
          IF (DUM.GE.AAMAX) THEN
            IMAX=I
            AAMAX=DUM
          ENDIF
16      CONTINUE
        IF (J.NE.IMAX)THEN
          DO 17 K=1,N
            DUM=A(IMAX,K)
            A(IMAX,K)=A(J,K)
            A(J,K)=DUM
17        CONTINUE
          D=-D
          VV(IMAX)=VV(J)
        ENDIF
        INDX(J)=IMAX
        IF(J.NE.N)THEN
          IF(A(J,J).EQ.0.)A(J,J)=TINY
          DUM=1./A(J,J)
          DO 18 I=J+1,N
            A(I,J)=A(I,J)*DUM
18        CONTINUE
        ENDIF
19    CONTINUE
      IF(A(N,N).EQ.0.)A(N,N)=TINY
      RETURN
      END

C-----------------------------------------------------------------------
      SUBROUTINE LUBKSB(A,N,NP,INDX,B)
C-----------------------------------------------------------------------
      DIMENSION A(NP,NP),INDX(N),B(N)
c
      II=0
      DO 12 I=1,N
        LL=INDX(I)
        SUM=B(LL)
        B(LL)=B(I)
        IF (II.NE.0)THEN
          DO 11 J=II,I-1
            SUM=SUM-A(I,J)*B(J)
11        CONTINUE
        ELSE IF (SUM.NE.0.) THEN
          II=I
        ENDIF
        B(I)=SUM
12    CONTINUE
      DO 14 I=N,1,-1
        SUM=B(I)
        IF(I.LT.N)THEN
          DO 13 J=I+1,N
            SUM=SUM-A(I,J)*B(J)
13        CONTINUE
        ENDIF
        B(I)=SUM/A(I,I)
14    CONTINUE
      RETURN
      END


C-----------------------------------------------------------------------
      REAL*4 FUNCTION erff(X)
c fast version of erf(x), using lookup tables
c added by J.S.
C-----------------------------------------------------------------------
      implicit none
      integer*4 mt
      real*4 xmax,dx,X
      parameter (mt=1024,xmax=5.0)
      parameter (dx=xmax/mt)
      real*4 erftab(0:mt),a,b,XX,SS,Z
      real*4 myerf
      integer*4 init,NX,i
      DATA init/0/
      SAVE erftab
      
      if (init.eq.0) then
        erftab(0)=0.0
	do i=1,mt
          erftab(i)=myerf(i*dx)
	enddo
	init=1
      endif
      
      XX=ABS(X)
      if (XX.LE.XMAX) THEN
        SS=SIGN(1.0,X)
	Z=XX/dx
        NX=NINT(Z)
	if (NX.lt.1) NX=NX+1
	if (NX.ge.mt) NX=NX-1         
        a=(erftab(nx+1)+erftab(nx-1))/2.0-erftab(nx)
	b=(erftab(nx+1)-erftab(nx+1))/2.0
	erff=SS*(erftab(nx)+a*(Z-nx)**2+b*(Z-nx)) 	      
      ELSE
        erff=1.0      
      ENDIF

      RETURN
      END
c
C $Log: res_exci_bcm.f,v $
C Revision 1.10  2006/05/11 16:33:10  saroun
C debugging bcm and phon models
C
C Revision 1.9  2006/05/10 18:46:32  saroun
C *** empty log message ***
C
C Revision 1.8  2006/05/08 21:39:57  saroun
C x   change BCM model: optional width profile as damped oscillators
C x   BUG FIX: CFG prompt does not show current as default
C x   BUG FIX: po navratu z MFIT, MAPSQ se vypne ray-tracing (MFIT->FIT)
C x   BUG FIX: subsequent print shows ellipsoid instead of  cloud
C x   PLOT jenom resol. funkce + disp. surface (bez dat)
C x   BUG FIX: SQOm, kdyz neni zadan komentar, zobrazi se divne znaky
C
                            
