/* 
$Id: excihandle.c,v 1.10 2006/05/02 16:29:41 saroun Exp $

C-wrappers for dynamically linked procedures (EXCI module) of RESTRAX
(c) J. Saroun, 2005, saroun@ujf.cas.cz
*/

#include <stdio.h>
#include <dlfcn.h>
#include <string.h>

#include "exci.h"


static char *lastname=NULL;
static float excinumber=0.0;
void *excihandle=NULL;
void (*exc)();
void (*readexcpar)();
void (*repexcpar)();
void (*openexcifile)();
void (*closeexcifile)();
void (*setexcidefault)();
float (*exciver)();

struct qomega *rqomega=NULL;
struct model *rmodel=NULL;
struct reclattice *rlat=NULL;


/* Copy string src to lastname, takes care about memory allocation
 returns number of copied characters */
int cpstring(const char *src) {
  char *cpt;
  if (lastname != NULL) {
    free(lastname);
    lastname=NULL;
  };
  if (src == NULL) {
    lastname=NULL;
    return 0;
  };
  
  lastname = (char *) malloc(strlen(src) + 1);
  if (lastname == NULL) return -1;
  cpt=strcpy(lastname,src);
  if (cpt == NULL) {
        printf("Error in strcpy: <%s> <%s>\n",lastname,src);
        return -2;        
  };
  return strlen(lastname);
};


/* Release exci library */
void releaseexci_() {
    char *error;
    if (excihandle) {
      dlerror();    /* Clear any existing error */
      dlclose(excihandle);
      if ((error = dlerror()) != NULL)  {
         fprintf (stderr, "%s\n", error);
         return;
      };               
      printf("EXCI released: %s\n",lastname);        
      if (lastname != NULL) {
        free(lastname);
        lastname=NULL;
      };
      excihandle=NULL;
      exc=NULL;
      readexcpar=NULL;
      repexcpar=NULL;
      openexcifile=NULL;
      setexcidefault=NULL;
      exciver=NULL;
      /* allocate data locally, so that they can be used by RESTRAX */
      rqomega=NULL;
      rmodel=NULL;
      rlat=NULL;     
    };     
};

/* Load symbol from EXCI, type void */
void *loadsym(const char *name, int *ires) {
  char *error;
  void *sh=NULL;
  dlerror();    /* Clear any existing error */
  if (*ires >= 0 && excihandle != NULL ) {
    *(void **) (&sh) = dlsym(excihandle, name);
    if ((error = dlerror()) != NULL)  {
      fprintf (stderr, "Error %s:\n cannot read symbol %s from %s \n", error,name,lastname);
      releaseexci_();
      *ires=-4;
    } else {
    *ires=1;
/*    printf("Loaded symbol: %s %p\n",name,sh);*/

    };
  };
  return sh;
};


/* Load the exci library dynamically 
Return: 
<0 if error occured (EXCI released)
=0 if nothing changed
=1 if new EXCI was loaded

*/
int loadexci_(char *excilibname) {
    int ires;
    int icom;
    double *q[4],*omexc[6],*sqom[6];
    long ilen,ilen1;


/* excilibname=="" => free previously loaded libraray and exit*/    
    if (excilibname=="") {
      releaseexci_();
      return -1;      
    };
    
/* compare new library name with the current one */ 
    if (lastname != NULL) {
      icom=strcmp(lastname,excilibname);
    } else { icom=-1; };
    
/* Load a new library named *excilibname if 
 (a) there is no library loaded, or  
 (b) the names of the current and requested libraries are different, or
 (c) excihandle == NULL 
*/ 
    if ((icom!=0) || (!excihandle)) {

/* release previously loaded libraray if necessary */    
      releaseexci_();    
      excihandle = dlopen (excilibname, RTLD_LAZY);
      if (!excihandle) {
        fprintf (stderr, "%s\n", dlerror());       
        releaseexci_();
        return -2;
      };     
      
      ilen=cpstring(excilibname);
      ires=0;              
/* load symbols for the fortran subroutines in exci */   
      *(void **) (&exciver)=loadsym("exciversion_",&ires);
      if (ires > 0) {
        excinumber=exciver();       
        printf("EXCI loaded: %s, version=%1.3f\n",lastname,excinumber);
      };
      
      *(void **) (&exc)=loadsym("exci_",&ires); 
      *(void **) (&readexcpar)=loadsym("readexcipar_",&ires); 
      *(void **) (&openexcifile)=loadsym("openexcifile_",&ires); 
      *(void **) (&closeexcifile)=loadsym("closeexcifile_",&ires); 
      *(void **) (&repexcpar)=loadsym("repexcipar_",&ires); 
      *(void **) (&setexcidefault)=loadsym("setexcidefault_",&ires);
/* load pointers to structures equivalent to the COMMON blocks in EXCI */   
      *(void **) (&rqomega)=loadsym("qomega_",&ires);
      *(void **) (&rmodel)=loadsym("model_",&ires);
      *(void **) (&rlat)=loadsym("reclattice_",&ires);
      
/* if everything is OK, set default values and initialize EXCI  */
      if (ires > 0) {
        (*setexcidefault)(); 

/* calling exci with icom<-10 will set user-defined defaults 
like parameter names, user data filenames, and other initializations
needed only once at library loading*/            
        icom=-999; 
        (*exc)(&icom,q,omexc,sqom); 
        return 1;
      } else return ires;      
    };
    return 0;
};


/* C wrapper for dynamically loaded fortran subroutines defined in shared library res_exci..*/

void exci_(int *icom, double *q,double *omexc,double *sqom)
{
  if (exc !=NULL ) (*exc)(icom,q,omexc,sqom);
};

void readexcipar_() 
{
 if (readexcpar !=NULL ) (*readexcpar)();
};

void repexcipar_() 
{
 if (repexcpar !=NULL ) (*repexcpar)();
};

void openexcifile_(int *iunit,char *fnam,int *ires) 
{
 if (openexcifile !=NULL ) (*openexcifile)(iunit,fnam,ires,strlen(fnam));
}; 

void closeexcifile_(int *iunit) 
{
 if (closeexcifile !=NULL ) (*closeexcifile)(iunit);
}; 

void setexcidefault_() {
 if (setexcidefault !=NULL ) (*setexcidefault)();
}; 


float exciversion_() 
{
 return excinumber;
};

void getmodel_(struct model *arg) {
 if (rmodel !=NULL ) *arg=*rmodel;
};

void setmodel_(struct model *arg) {
 if (rmodel !=NULL ) *rmodel=*arg;
};

void getqomega_(struct qomega *arg) {
 if (rqomega !=NULL ) *arg=*rqomega;
};

void setqomega_(struct qomega *arg) {
 if (rqomega !=NULL ) *rqomega=*arg;
};

void getreclat_(struct reclattice *arg) {
 if (rlat !=NULL ) *arg=*rlat;
};

void setreclat_(struct reclattice *arg) {
 if (rlat !=NULL ) *rlat=*arg;
};

/*  light version of getqomega ... does not pass arrays*/
void getqomegainf_(int *ndatqom, double *chkqom) {
 if (rqomega !=NULL ) {
   *ndatqom=(*rqomega).ndatqom;
   *chkqom=(*rqomega).chkqom;
 };
};

/*  light version of setqomega ... does not pass arrays*/
void setqomegainf_(int *ndatqom, double *chkqom) {
 if (rqomega !=NULL ) {
   (*rqomega).ndatqom=*ndatqom;
   (*rqomega).chkqom=*chkqom;
 };
};
