C//////////////////////////////////////////////////////////////////////
C////  $Id: res_cal.f,v 1.5 2006/05/09 18:05:06 saroun Exp $
C////
C////  R E S T R A X   4.4
C////
C////  Transformations between coordinate systems etc.
C////
C////  
C//////////////////////////////////////////////////////////////////////


C--------------------------------------------------------------------
      SUBROUTINE RECLAT
C  Calculate transformation matrices for reciprocal lattice
C
C  Input: 
C  AQ(3)    ... unit cell base vectors
C  ALFA(3) ... unit cell angles
C  A1(3), A2(3) ... perpendicular vectors in scattering plane 
C
C  Output:
C  SMAT(i,j) ... transforms Q(hkl) from r.l.u. to 
C  the orthogonal coordinates given by base vectors A1(i), A2(i), A3(i) 
C  COSB(3) ... direction cosines for unit cell base vectors
C  revised 22/2/2005, J.S.
C--------------------------------------------------------------------
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'lattice.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'exciimp.inc'
      
      RECORD /RECLATTICE/ rl
      
      REAL*8 EPS,ZD,RD,XCC,C1,C2,C3
      INTEGER*4 I,J,K,L,M,IER

      PARAMETER(EPS=1.0D-8)
      REAL*8 AQ(3),ALFA(3),A1(3),A2(3)
      REAL*8 A(3),XBB(3,3),V1(3),V2(3),V3(3),U(3,3),COSA(3),SINA(3),SINB(3)
c      REAL*8 W1(3),W2(3),W3(3),XBBI(3,3),
      REAL*8 A3(3),AUX(3,3),ZN(4),DUM
      REAL*8 S(3,3),SD(3,3),B(3)
      COMMON /ERROR/IER
c      REAL*8 QxQ 
       
      EQUIVALENCE (AQ(1),RES_DAT(i_AS))
      EQUIVALENCE (ALFA(1),RES_DAT(i_AA))
      EQUIVALENCE (A1(1),RES_DAT(i_AX))
      EQUIVALENCE (A2(1),RES_DAT(i_BX))
     
c      write(*,*) 'RECLAT BEFORE:'
10    format('res_cal.f, RECLAT, ',a,1x,10(1x,G10.4))
       
c1000  format(a15,3(2x,E10.3))
       IER=0
       ZD=2.D0*PI
       RD=PI/180.D0
       XCC=0.D0
       DO I=1,3
        A(I)=AQ(I)/ZD
        IF(ABS(A(I)).LT..00001) THEN  ! check lattice spacing
           IER=1
           write(sout,30)
           RETURN
        ENDIF  
        COSA(I)=COS(ALFA(I)*RD)
        SINA(I)=SIN(ALFA(I)*RD)
        XCC=XCC+COSA(I)**2
      ENDDO
      XCC=1.+2.*COSA(1)*COSA(2)*COSA(3)-XCC
      IF(XCC.LE.0.) THEN    ! check lattice angles
        IER=2
        write(sout,31)
        RETURN
      ENDIF  
         
         
      XCC=SQRT(XCC)
      CELLVOL=XCC*AQ(1)*AQ(2)*AQ(3)   ! this is unit cell volume
      J=2
      K=3
      DO I=1,3
        B(I)=SINA(I)/(A(I)*XCC)            ! length of the r.l. axes an Ang1
        COSB(I)=(COSA(J)*COSA(K)-COSA(I))/(SINA(J)*SINA(K))
        SINB(I)=SQRT(1.-COSB(I)*COSB(I))  ! angles btw. r.l. axes
        J=K
        K=I
      ENDDO



c check that A,B are perpendicular
c NOTE: QxQ uses only COSB, not SMAT ...
c       IF (ABS(QxQ(A1(1),A2(1))).GT.EPS) THEN
c          write(sout,33)          
c        ENDIF
    
C  SD(i,j) is a matrix defined so that
C  Qhkl = SQRT((hkl)*SD*(hkl))   (in A^-1)

        DO I=1,3   
           SD(I,I)=B(I)**2
        END DO
        SD(1,2)=(COSA(1)*COSA(2)-COSA(3))/AQ(1)/AQ(2)*(2*PI/XCC)**2
        SD(1,3)=(COSA(1)*COSA(3)-COSA(2))/AQ(1)/AQ(3)*(2*PI/XCC)**2
        SD(2,3)=(COSA(2)*COSA(3)-COSA(1))/AQ(2)/AQ(3)*(2*PI/XCC)**2
        SD(2,1)=SD(1,2)
        SD(3,1)=SD(1,3)
        SD(3,2)=SD(2,3)    
    

C XBB(i,j) are projections of rec. lat. base vectors on the orthonormal base:
C Let (a,b,c) and (a*,b*,c*) are direct and reciprocal lattice base vectors
C assume a parallel to a*
C XBB(i,1) ... parallel to a*
C XBB(i,2) ... parallel to c x a*
C XBB(i,3) ... parallel to c 
C i.e. the columns are a*,b*,c* in the new orthonormal base 
  
        XBB(1,1)=B(1)
        XBB(2,1)=0.D0
        XBB(3,1)=0.D0
        XBB(1,2)=B(2)*COSB(3)
        XBB(2,2)=B(2)*SINB(3)
        XBB(3,2)=0.D0
        XBB(1,3)=B(3)*COSB(2)
        XBB(2,3)=-B(3)*SINB(2)*COSA(1)
        XBB(3,3)=1/A(3)                   
      
c        CALL INVERT(3,XBB,3,XBBI,3)            

C convert A1,A2 to the new orthonormal system:       
      DO I=1,3
        V1(I)=0.D0
        V2(I)=0.D0
        DO  J=1,3
          V1(I)=V1(I)+XBB(I,J)*A1(J)
          V2(I)=V2(I)+XBB(I,J)*A2(J)
        ENDDO
      ENDDO
C get V3 perpendicular to V1,V2
      V3(1)=V1(2)*V2(3)-V1(3)*V2(2)
      V3(2)=V1(3)*V2(1)-V1(1)*V2(3)
      V3(3)=V1(1)*V2(2)-V1(2)*V2(1)
C get V2 perpendicular to V3,V1
      V2(1)=V3(2)*V1(3)-V3(3)*V1(2)
      V2(2)=V3(3)*V1(1)-V3(1)*V1(3)
      V2(3)=V3(1)*V1(2)-V3(2)*V1(1)
C get norms of V1,V2,V3
      C1=V1(1)**2+V1(2)**2+V1(3)**2
      C2=V2(1)**2+V2(2)**2+V2(3)**2
      C3=V3(1)**2+V3(2)**2+V3(3)**2
      C1=SQRT(C1)
      C2=SQRT(C2)
      C3=SQRT(C3)
       
C convert V1,V2 back to r.l.:       
c       DO I=1,3
c         W1(I)=0.D0
c         W2(I)=0.D0
c         W3(I)=0.D0
c         DO  J=1,3
c           W1(I)=W1(I)+XBBI(I,J)*V1(J)
c            W2(I)=W2(I)+XBBI(I,J)*V2(J)
c            W3(I)=W3(I)+XBBI(I,J)*V3(J)
c          ENDDO
c        ENDDO
       
      
      IF (C1*C2*C3.LT.EPS) THEN ! check scattering plane
        IER=3
        WRITE(SOUT,32)
        RETURN
      ENDIF
       
C U(i,j) is the orthonormal system made from V1,V2,V3 (also called AB system in RESTRAX)
       DO I=1,3
          U(1,I)=V1(I)/C1
          U(2,I)=V2(I)/C2
           U(3,I)=V3(I)/C3
       ENDDO
       DO K=1,3
         DO M=1,3
           S(K,M)=0.D0
           DO L=1,3
               S(K,M)=S(K,M)+U(K,L)*XBB(L,M)
           ENDDO
          ENDDO
        ENDDO 
                          
C SMAT converts Q from r.l.u. to AB coordinates (in A^-1)
C SINV is inverse to SMAT
C the matrices are 4-dimensional so that they can operate in (h,k,l,energy) space
      CALL INVERT(3,S,3,AUX,3)            
      DO I=1,4
      DO J=1,4         
        IF(I.LE.3.AND.J.LE.3) THEN
          SMAT(I,J)=S(I,J)
          SINV(I,J)=AUX(I,J)
        ELSE IF (I.EQ.4.AND.J.EQ.4) THEN
          SMAT(I,J)=1.D0
          SINV(I,J)=1.D0
        ELSE
          SMAT(I,J)=0.D0
          SINV(I,J)=0.D0
       ENDIF
      ENDDO
      ENDDO  
c      write(*,10) 'SMAT: ',SMAT(1:3,1)
c      write(*,10) 'SMAT: ',SMAT(1:3,2)
c      write(*,10) 'SMAT: ',SMAT(1:3,3)
c      write(*,10) 'SINV: ',SINV(1:3,1)
c      write(*,10) 'SINV: ',SINV(1:3,2)
c      write(*,10) 'SINV: ',SINV(1:3,3)
c      pause

C// MABR converts Q from r.l.u. to A,B coordinates, taking A,B as base vectors
C// MRAB is inverse to MABR
      
      DO I=1,3
        A3(I)=SINV(I,3)*C1
      ENDDO    
                     
      CALL QNORM(A1,DUM,ZN(1))  ! get norms of vectors A,B,C in Ang^-1
      CALL QNORM(A2,DUM,ZN(2))
      CALL QNORM(A3,DUM,ZN(3))      
      
      ZN(4)=1.D0
      DO I=1,4               
       DO J=1,4
          MRAB(I,J)=SINV(I,J)*ZN(J)
          MABR(I,J)=SMAT(I,J)/ZN(I)
       ENDDO
      ENDDO
  
c pass rec. lattice parameters to EXCI  

      DO i=1,3
        rl.cell(i)=AQ(i)
        rl.cell(i+3)=ALFA(i)
        rl.VECA(i)=A1(i)
        rl.VECB(i)=A2(i) 
        rl.cosb(i)=COSb(i)        
      ENDDO
      DO i=1,4
        DO j=1,4
         rl.smat(i,j)=smat(i,j)
         rl.sinv(i,j)=sinv(i,j)
         rl.mrab(i,j)=mrab(i,j)
         rl.mabr(i,j)=mabr(i,j)
        ENDDO
      ENDDO
      rl.cellvol=cellvol 
      call setreclat(rl)
  
  
      RETURN
30    FORMAT(' RECLAT: Check Lattice Spacings (AS,BS,CS)')
31    FORMAT(' RECLAT: Check Cell Angles (AA,BB,XCC)')
32    FORMAT(' RECLAT: Check Scattering Plane (AX....BZ)')
33    FORMAT(' RECLAT: Vectors A,B are not perpendicular !')
      END


C--------------------------------------------------------------------
      SUBROUTINE RLU2AB(A,AB)
C Converts A matrix from r.l.u. to coordinates given by
C the base vectors AX..BZ in the scat. plane (assumed orthogonal)
C--------------------------------------------------------------------
      IMPLICIT NONE
        
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'lattice.inc'
      REAL*8 A(4,4),AB(4,4)

      CALL BTAB4(A,MRAB,AB)

      END  

C***********************************************************************
      SUBROUTINE CN2RLU(A,AR)
C  Converts A from Cooper&Nathans to recip. lattice coordinates
C***********************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
       
      REAL*8 A(4,4),AR(4,4)
      CALL BTAB4(A,MCR,AR)
      END       
        

C***********************************************************************
      SUBROUTINE CN2RLU_MF(A,AR,ITEM)
C  Version of CN2RLU for ITEM-th data set
C***********************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'restrax.inc'
       
      REAL*8 A(4,4),AR(4,4)
      INTEGER*4 ITEM   
      CALL BTAB4(A,mf_MCR(1,1,ITEM),AR)
      END       


        
C--------------------------------------------------------------------
      SUBROUTINE GET_SCANGLES(KI,KF,Q,SS,OM,PSI,IER)
C  Calculates scattering angle (OM) and angle between Q and KI (PSI)
C  Input is KI,KF,Q,SS, which determine the scattering triangle
C  IER=2 ... error in getting angle, cannot close triangle 
C--------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      
      REAL*8 KI,KF,Q,SS,OM,PSI
      INTEGER*4 IER
      REAL*8 CO
      
      IER=2
      IF (ABS(Q).LT.1E-6) THEN
        OM=0.
        PSI=PI/2.
        IER=0
        RETURN      
      ENDIF
C///  calculates scattering angle omega=2*thetaS
      CO=(KI**2+KF**2-Q**2)/(2*KI*KF)
      IF(ABS(CO).GT.1) GOTO 99
      OM=SIGN(1.D0,SS)*ABS(ACOS(CO))

C///  calculates scattering angle omega=2*thetaS      
      CO=(KF**2-KI**2-Q**2)/(2*KI*Q)
      IF(ABS(CO).GT.1) GOTO 99
      PSI=SIGN(1.D0,SS)*ABS(ACOS(CO))
      IER=0
      RETURN
 
99    RETURN
      END 
 
 
C----------------------------------------------------------------------------- 
      SUBROUTINE ANGSCAN(DA3,DA4)
C set a scan in QHKL,E equivalent to angular scan (small range) by da3 or da4       
C----------------------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      REAL*8 A3,A4,DA3,DA4,VQ(4),WQ(4),SI,CO
      INTEGER*4 I
1     format(' (DH,DK,DL,EN) = ',4(2x,F7.4))
2     format(' VQ = ',4(2x,F7.4))

c      write(*,*) 'AGSCAN: ',DA3,DA4
      IF(DA3.NE.0.OR.DA4.NE.0) THEN
        A4=DA4*PI/180
        SI=sin(A4)
        CO=cos(A4)
c VQ is the change in QHKL due to rotation by A4, Lab. coordinates (z || ki, y vertical)
        VQ(1)=KF0*(COMEGA*SI+SOMEGA*(CO-1.D0))
        VQ(3)=KF0*(-SOMEGA*SI+COMEGA*(CO-1.D0))
        VQ(2)=0.
        VQ(4)=0.
c      write(*,2) (VQ(I), I=1,4)
        CALL MXV(-1,4,4,MLC,VQ,WQ)  ! from Lab to CN coord.
c      write(*,2) (VQ(I), I=1,4)
c add an increment due to sample rotation (dA3)
        A3=DA3*PI/180
        SI=SIN(A3)
        CO=COS(A3)
        WQ(1)=WQ(1)+Q0*(CO-1.D0)
        WQ(2)=WQ(2)-Q0*SI
c      write(*,2) (VQ(I), I=1,4)
c convert from CN to r.l.u.
        CALL MXV(1,4,4,MRC,WQ,VQ)   
c      write(*,2) (VQ(I), I=1,4)
        DO I=1,4
         RES_DAT(i_DQH+I-1)=VQ(I)
        ENDDO
      ENDIF
      END

C*********************************************************************************
      SUBROUTINE ROTA3(QI,A3,QF)
C transform QI(3) to QF(3) through the rotation by axis A3
C Q are given in r.l.u.
C WARNING! QI is REAL*4 !!
C*********************************************************************************
      IMPLICIT NONE
      
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'lattice.inc'
      
      INTEGER*4 I,J
      REAL*8 A3,QF(3)
      REAL*4 QI(3)
      REAL*8 DA3 ,VQ(3),WQ(2),SI,CO
      
C transform from r.l.u to AX..BZ      
      DO J=1,3
        VQ(J)=0.D0
      ENDDO   
      DO I=1,3
        DO J=1,3
          VQ(J)=VQ(J)+SMAT(J,I)*QI(I)
        ENDDO   
      ENDDO   
C rotate by A3 around vertical axis (i=3)
      DA3=A3*deg
      SI=SIN(DA3)
      CO=COS(DA3)
      WQ(1)=VQ(1)*CO+VQ(2)*SI
      WQ(2)=VQ(2)*CO-VQ(1)*SI
C transform back from AX..BZ to r.l.u      
      DO I=1,3
c         QF(I)=SINV(1,I)*WQ(1)+SINV(2,I)*WQ(2)
         QF(I)=SINV(I,1)*WQ(1)+SINV(I,2)*WQ(2)+SINV(I,3)*VQ(3)
      ENDDO
      END

C------------------------------------------------------------------------------------
      SUBROUTINE ROTA4(A4,Q)
C Increment scattering angle by A4 [rad]
C Only changes QHKL, DOESN'T UPDATE DEPENDENT FIELDS !!
C------------------------------------------------------------------------------------ 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      REAL*8 A4,VQ(4),WQ(4),Q(3),SI,CO
      INTEGER*4 I

      IF(A4.NE.0.) THEN
        SI=sin(A4)
        CO=cos(A4)
C// VQ is the change of Q in Lab. coordinates        
        VQ(1)=KF0*(COMEGA*SI+SOMEGA*(CO-1.0))
        VQ(3)=KF0*(COMEGA*(CO-1.0)-SOMEGA*SI)
        VQ(2)=0.
        VQ(4)=0.
        CALL MXV(-1,4,4,MLC,VQ,WQ)  ! from Lab to CN coord.
        CALL MXV(1,4,4,MRC,WQ,VQ)   ! from CN to r.l.u.
C// Increment QHKL        
        DO I=1,3
         RES_DAT(i_QH+I-1)=RES_DAT(i_QH+I-1)+VQ(I)
        ENDDO
C swith SS to -SS if needed        
        IF ((OMEGA+A4)*OMEGA.LE.0) RES_DAT(i_SS)=-RES_DAT(i_SS) 
C        CALL BEFORE ! don't call BEFORE
      ENDIF
      END
       
C--------------------------------------------------------------------
      SUBROUTINE GET_A3A4(ID,Q,A3,A4,IER)
C returns A3,A4 angles for given Q(4). 
C ID-th data set is taken as reference (where A3=0)
C if IR>0, error has occured (cannot construct the triangle)
C--------------------------------------------------------------------
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      INTEGER*4 ID,IER
      REAL*8 Q(4),A3,A4
      REAL*8 VKI,VKF,QQ
      REAL*8 KFIX,KOM,DUM,PSI,PSIref,A4ref,Qref
             
      IER=1
C Get correct KI,KF,Q values 
      KOM=Q(4)/HSQOV2M
      CALL QNORM(Q(1),DUM,QQ)
      KFIX=mf_par(I_KFIX,ID)
      IF (mf_par(i_FX,ID).EQ.1) THEN 
         VKI=KFIX
         VKF=KFIX**2-KOM
         IF (VKF.LE.0) GOTO 99
         VKF=SQRT(VKF)
      ELSE
         VKF=kfix
         VKI=kfix**2+kom   
         IF (VKI.LE.0) GOTO 99
         VKI=SQRT(VKI)
      ENDIF
                  
C///  calculates scattering angle (A4) and angle between Ki and Q (PSI)
      CALL QNORM(mf_par(i_QH,ID),DUM,Qref)
      CALL GET_SCANGLES(VKI,VKF,Qref,mf_par(i_SS,ID),A4ref,PSIref,IER) 
      IF (IER.NE.0) GOTO 99
      CALL GET_SCANGLES(VKI,VKF,QQ,mf_par(i_SS,ID),A4,PSI,IER) 
      IF (IER.NE.0) GOTO 99

c11    format(a,6(1x,G12.6))      
c      write(*,*) 'Qref, Q: ',Qref,Q
c      write(*,*) 'REF A4, PSI: ',A4ref/deg,PSIref/deg
c      write(*,*) 'CUR A4, PSI: ',A4/deg,PSI/deg

C///  calculates sample rotation with respect to 
C///  the reference channel (=ID)
      IER=3
C// angle between Q and reference (=QHKL from IDth channel)      
      CALL GET_ANGLE(mf_par(i_QH,ID),Q(1),A3)
c      write(*,*) 'Q vs. Qref: ',A3/deg
      A3=PSI-PSIref-A3      
      
c      call M4xV4_3(mf_MCR(1,1,ID),Q,VQ) ! convert QHKL to C&N
c      CA3=VQ(1)/QQ
c      SA3=VQ(2)/QQ
c      A3=SIGN(1.D0,SA3)*ACOS(CA3)
            
      IER=0
99    RETURN
      END      

        
C--------------------------------------------------------------------
      SUBROUTINE SCATTRIANGLE
C  - define scatteing triangle and associated fields
C  - create transformation matrices between C&N and rec. lattice (MCR,MRC)
C  - create transformation matrix Lab. coord. -> C&N
C Lab. coordinates are defined with Z//Ki, Y vertical 
C W(er.lat.) = MRC*V(C&N)
C MCR = MRC^-1
C RECLAT must be called before !!!!
C--------------------------------------------------------------------
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'lattice.inc'
      
      REAL*8 SI,CO,KFIX,KOM,DUM,XQ,YQ,QQ,TRANS
      INTEGER*4 IER,I,J
      CHARACTER*40 MATER(3)
       
       DATA MATER /
     1'  Check scaterring triangle             ',
     2'  Check monochromator dhkl              ',
     3'  Check analyzer dhkl                   '/

600   FORMAT(a20,4(2X,E12.6))       
601   FORMAT(1x,'KI: ',E12.6,2x,'KF: ',E12.6,2x,'Q: ',E12.6)      
602   FORMAT(1x,'DM: ',E12.6,2x,'KI: ',E12.6,2x)      
603   FORMAT(1x,'DA: ',E12.6,2x,'KF: ',E12.6,2x)      
501   FORMAT(A)      

C Get correct KI,KF,Q values

      IER=1
C Get correct KI,KF,Q values
      KOM=RES_DAT(I_EN)/HSQOV2M
      CALL QNORM(RES_DAT(i_QH),DUM,Q0)
      KFIX=RES_DAT(I_KFIX)
      IF (RES_DAT(i_FX).EQ.1) THEN 
         KI0=KFIX
         KF0=KFIX**2-KOM
         KF0=SIGN(1.D0,KF0)*SQRT(ABS(KF0))
      ELSE
         KF0=kfix
         KI0=kfix**2+kom   
         KI0=SIGN(1.D0,KI0)*SQRT(ABS(KI0))
      ENDIF
      IF ((RES_DAT(i_SM).NE.0).AND.(KI0.LE.PI/RES_DAT(i_DM))) THEN
        IER=2
        GOTO 99
      ENDIF  
      IF ((RES_DAT(i_SA).NE.0).AND.(KF0.LE.PI/RES_DAT(i_DA))) THEN
        IER=3
        GOTO 99
      ENDIF  
C///  calculates scattering angle omega=2*thetaS
      COMEGA=(KI0**2+KF0**2-Q0**2)/(2*KI0*KF0)
      IF(ABS(COMEGA).GT.1) GOTO 99
      IF(RES_DAT(i_SS).GT.0) THEN
        SOMEGA=SQRT(1-COMEGA**2)
        OMEGA=ACOS(COMEGA)
      ELSE
        SOMEGA=-SQRT(1-COMEGA**2)
        OMEGA=-ACOS(COMEGA)
      ENDIF                    
      
C///  trans. matrix C&N  -->  Lab
      DO I=1,4
        DO J=1,4     
           MLC(I,J)=0.D0     
        ENDDO 
      ENDDO  
      MLC(4,4)=1.D0
C angle between Ki and Q
      CO=(KF0**2-KI0**2-Q0**2)/(2*KI0*Q0)
      IF(ABS(CO).GT.1) GOTO 99  ! should never happen !
      IF(RES_DAT(i_SS).GT.0) THEN 
        SI=SQRT(1.D0-CO**2)
      ELSE
        SI=-SQRT(1.D0-CO**2)
      ENDIF
      MLC(1,1)=SI
      MLC(1,2)=CO
      MLC(2,3)=1.D0
      MLC(3,1)=CO
      MLC(3,2)=-SI


C///  trans. matrix r.l.u. --> C&N

      XQ=TRANS(QHKL,1)
      YQ=TRANS(QHKL,2)                              
      QQ=SQRT(XQ**2+YQ**2)
      
      CO=XQ/QQ
      SI=YQ/QQ
      DO 11 I=1,3
       MCR(1,I)=SMAT(1,I)*CO+SMAT(2,I)*SI
       MCR(2,I)=SMAT(2,I)*CO-SMAT(1,I)*SI
       MCR(3,I)=SMAT(3,I)
       MCR(4,I)=0.
11       MCR(I,4)=0.
      MCR(4,4)=1.                       ! (MCR): r.l.u. --> C&N
      CALL INVERT(4,MCR,4,MRC,4)        ! (MRC): C&N    -->  r.l.u.                  
      IER=0 
      RETURN

99    CONTINUE
      write(sout,501) MATER(IER)
      if (ier.eq.1) write(sout,601) ki0,kf0,q0
      if (ier.eq.2) write(sout,602) RES_DAT(i_DM),ki0
      if (ier.eq.3) write(sout,603) RES_DAT(i_DA),kf0
      RETURN
      END

C--------------------------------------------------------------------
      SUBROUTINE TRANSMAT
C  creates transformation matrices for the four coordinate systems:     
C  R  ...  reciprocal lattice  
C  C  ...  Cooper & Nathans coordinates (C&N)
C  G  ...  C&N rotated so that X(1) // grad E(Q)
C  D  ...  G rotated so that X(4) // normal to the disp. surface    
C--------------------------------------------------------------------
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'

      REAL*8 MGD(4,4),PHI,ZRA,GNR,GNA 
      INTEGER*4 I,J       

      CALL GETMAT(GRD,MCG)              
      
! (MGD): grad   <--  disp      
      DO I=1,4
        DO J=1,4
          MGD(I,J)=0
        ENDDO
        MGD(I,I)=1
      ENDDO
      CALL QNORM(GRD,GNR,GNA)
      ZRA=GNR/GNA            
      PHI=ATAN(RES_DAT(i_GMOD)*ZRA)       ! units of GMOD are Energy/r.l.u.!
      MGD(1,1)=COS(PHI)
      MGD(4,4)=COS(PHI)
      MGD(1,4)=-SIN(PHI)
      MGD(4,1)=SIN(PHI)                        
! (MCD): C&N    <--  disp   
      CALL MXM(1,4,4,MCG,MGD,MCD)        
! (MDR): r.l.u. -->  disp
      CALL MXM(-1,4,4,MCD,MCR,MDR)      

      END

C--------------------------------------------------------------------
      SUBROUTINE GETMAT(V,MAT)
C  creates rotation matrix converting C&N coordinates to
C  coordinates with X//V 
C--------------------------------------------------------------------
      IMPLICIT NONE   

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'

      REAL*8 MAT(4,4),V(3) 
      REAL*8 XV,YV,ZV,XQ,YQ,VV,VV1,QQ,SI1,CO1, SI2,CO2,TRANS
      INTEGER*4 I,J
       
C V to AB coordinates
      XV=TRANS(V,1)
      YV=TRANS(V,2)
      ZV=TRANS(V,3)
C QHKL to AB coordinates
      XQ=TRANS(QHKL,1)
      YQ=TRANS(QHKL,2)                              
      VV=SQRT(XV**2+YV**2+ZV**2)
      VV1=SQRT(XV**2+YV**2)
      QQ=SQRT(XQ**2+YQ**2)


C///  Transformation from (V//x) to C&N
C special case: GRD vertical:
C define Gy//Qy:
      IF (ABS(VV1).LE.1E-7) THEN
        DO i=1,3
        DO j=1,3
          MAT(i,j)=0.D0
        ENDDO
        ENDDO
        MAT(3,1)=-1.D0
        MAT(2,2)=1.D0
        MAT(1,3)=1.D0
      ELSE
C sagital angle of V
        SI1=ZV/VV
        CO1=SQRT(1.D0-SI1**2)
C angle between Q and V(component in the scatt. plane)
C sign is positive from QHKL to V 
        SI2=(XQ*YV-YQ*XV)/QQ/VV1
        CO2=(XV*XQ+YV*YQ)/QQ/VV1
 
        MAT(1,1)= CO1*CO2
        MAT(2,1)= CO1*SI2
        MAT(3,1)= +SI1     
        MAT(1,2)= -SI2
        MAT(2,2)= CO2
        MAT(3,2)=0 
        MAT(1,3)=-SI1*CO2 
        MAT(2,3)=-SI1*SI2 
        MAT(3,3)=CO1
      ENDIF
      DO I=1,3
        MAT(I,4)=0
        MAT(4,I)=0
      ENDDO
      MAT(4,4)=1                        

      END

C
       SUBROUTINE GETNORMS(ICOM,VI,VF,VRES,R0PHON,R0BRAG)
C***********************************************************************
C   returns volumes of VI (ki), VF (kf) and VRES (Q,E)
C   returns normalizing factors for phonons and Bragg scans
C   R0PHON=VI*VF/VRES, R0BRAG=R0PHON*(Bragg width)
C
C***********************************************************************
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'trax.inc'
      INCLUDE 'restrax.inc'

      INTEGER*4 ICOM      
      REAL*8 VI,VF,VRES,R0PHON,R0BRAG
      REAL*8 WB,WV,WV1,DTM
      REAL*8 AUX1(4,4)
      REAL*8 DETERM

      IF (ICOM.EQ.1) THEN               ! TRAX
        VI=VOLCKI
        VF=VOLCKF     
c        CALL INVERT(4,ATRAX,4,AUX,4)
        DTM=DETERM(ATRAX,4,AUX1)
        CALL GETRESSIZE(ATRAX,QHKL,0,WB,WV,WV1)
      ELSE                              ! Monte Carlo
        VI=VKINESS
        VF=VKFNESS     
c        CALL INVERT(4,ANESS,4,AUX,4)
        DTM=DETERM(ANESS,4,AUX1)
        CALL GETRESSIZE(ANESS,QHKL,0,WB,WV,WV1)
      ENDIF      
      WB=WB*SQRT(2*PI)/SQRT8LN2
      IF (DTM.GT.0) THEN
        VRES=(2.*PII)**2/SQRT(DTM)  
        R0PHON=VI*VF/VRES
        R0BRAG=R0PHON*WB
      ELSE  
        VRES=0
        R0PHON=0
        R0BRAG=0
      ENDIF  
      END
      

C---------------------------------------------------------
      SUBROUTINE GETRESSIZE(A,DIR,UNI,BRAG,EVAN,VAN)
C     Returns Bragg and EVANadium (dE=0) fwhm in direction DIR 
C     Units:
c     A-1    (UNI=0) or 
C     r.l.u. (UNI=1) or
C     steps  (UNI=2) step is defined by DIR in r.l.u.
C---------------------------------------------------------
       IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
       INCLUDE 'rescal.inc'
       REAL*8 C8LN2
       PARAMETER (C8LN2=5.545177444)
       REAL*8 MAT(4,4),A(4,4),AUX(4,4),A1(4,4),DIR(3)
       REAL*8 XRL,XCN,BRAG,EVAN,VAN,GETFWHM
       INTEGER*4 UNI

       CALL QNORM(DIR,XRL,XCN)
       
       IF (XRL.GT.0) THEN
       
       CALL GETMAT(DIR,MAT)
       CALL MXM(1,4,4,A,MAT,AUX)      
       CALL MXM(-1,4,4,MAT,AUX,A1) 
       
       CALL INVERT(4,A1,4,AUX,4)           
       VAN=SQRT(C8LN2*AUX(1,1))
       EVAN=GETFWHM(A1,1)
       BRAG=SQRT(C8LN2/A1(1,1))
       IF(UNI.EQ.1) THEN
         CALL QNORM(DIR,XRL,XCN)
         VAN=VAN*XRL/XCN
         EVAN=EVAN*XRL/XCN
         BRAG=BRAG*XRL/XCN
       ELSE IF (UNI.EQ.2) THEN
         CALL QNORM(DIR,XRL,XCN)
         VAN=EVAN/XCN
         EVAN=EVAN/XCN
         BRAG=BRAG/XCN
       ENDIF
       
       ELSE
         EVAN=0.
         BRAG=0.          
         VAN=0.          
       ENDIF
       
       END   

C----------------------------------------------------
      SUBROUTINE GETPHONWIDTH(ARES,WA,WR,WS)
C// returns phonon fwhm in [Energy]x[A-1](WA) , [r.l.u] (WR)
C//  and [steps] (WS)
C----------------------------------------------------
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      
      REAL*8 A(4,4),AUX(4,4),ARES(4,4),SCD(4),WA,WR,WS,DNR,DNA
                          
! scan direction to coordinates with axis 4 normal to the disp. surface (D-coordinates)
      CALL MXV(1,4,4,MDR,DELQ,SCD)  
      CALL QNORM(DELQ,DNR,DNA) 
      DNR=SQRT(DNR**2+RES_DAT(i_DEN)**2)
      DNA=SQRT(DNA**2+RES_DAT(i_DEN)**2)              
      IF(ABS(SCD(4)).LT.1.E-8) GOTO 999         
      CALL MXM(1,4,4,ARES,MCD,AUX)
      CALL MXM(-1,4,4,MCD,AUX,A)               ! Res. matrix to D-coordinates
      CALL INVERT(4,A,4,AUX,4)
      WS=SQRT(AUX(4,4))*SQRT8LN2/ABS(SCD(4))     ! fwhm in steps
      WA=WS*DNA
      WR=WS*DNR
      RETURN
      
999   WA=1.D10
      WR=1.D10
      WS=1.D10      
      
      END



C--------------------------------------------------------------------
      SUBROUTINE GetProj(A,QE,A1,QE1,MCN)
C
C     Transforms the resolution matrix A and vector QE to A1 and QE1,
C     in the coordinates with X(1) parallel to grad(E) and scaled in R.L.U.
C     MCN transforms vectors(Q,E) to the same coordinates
C     (!! axes 2,3 are not transformed  !!)
C--------------------------------------------------------------------
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'

      REAL*8 A(4,4),A1(4,4),QE(4),QE1(4)
      REAL*8 AUX(4,4),MCN(4,4)
      REAL*8 ZRA,GNR,GNA
      INTEGER*4 i,j

      CALL QNORM(GRD,GNR,GNA)
      ZRA=GNR/GNA

      CALL MXM(1,4,4,A,MCG,AUX)
      CALL MXM(-1,4,4,MCG,AUX,A1)

      do i=1,4
        A1(1,i)=A1(1,i)/ZRA
        A1(i,1)=A1(1,i)
      end do
      A1(1,1)=A1(1,1)/ZRA

      DO 10 i=1,4
      DO 10 j=1,4
         MCN(i,j)=MCG(i,j)
         if(j.eq.1) MCN(i,j)=MCN(i,j)*ZRA
10    CONTINUE

      CALL MXV(-1,4,4,MCN,QE,QE1)


      END

C***********************************************************************
      SUBROUTINE FWHM(ICOM) 
C writes fwhm of section and projection in arbitrary direction  
C (i.e. Bragg width and 'vanadium width' at dE=0) 
C *** J.S. February 1999  
C ICOM=1 .. TRAX
C ICOM=2 .. Monte Carlo (added by J.S. Sept. 2002)   
C***********************************************************************

      IMPLICIT NONE
       
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'
       
      REAL*8 BRLU,BANG,VRLU,VANG,V(3),WA,WR
      INTEGER*4 I,ICOM
900   FORMAT('Analytical result  (TRAX):')
901   FORMAT('Monte Carlo result (NESS):')
       
1     FORMAT(1x,'Bragg width:                   ',G12.5,' [A-1]',
     *       3x,G12.5,' [r.l.u]')
2     FORMAT(1x,'Elastic ''Vanad'' width (dE=0): ',G12.5,' [A-1]',
     *       3x,G12.5,' [r.l.u]')
3     FORMAT(1x,'Inelastic ''Vanad'' width:      ',G12.5,' [A-1]',
     *       3x,G12.5,' [r.l.u]')
      IF (NOS.GE.3) THEN
        IF (ABS(RET(1))+ABS(RET(2))+ABS(RET(3)).LE.1.D-30) THEN
          WRITE(SOUT,*) 'Direction vector has zero length !'
            RETURN
        ENDIF 
        DO I=1,3
           V(I)=RET(I)
        ENDDO
        IF (ICOM.EQ.1) THEN
          CALL GETRESSIZE(ATRAX,V,0,BANG,VANG,WA)
          CALL GETRESSIZE(ATRAX,V,1,BRLU,VRLU,WR)
          write(sout,900)
        ELSE
          CALL GETRESSIZE(ANESS,V,0,BANG,VANG,WA)
          CALL GETRESSIZE(ANESS,V,1,BRLU,VRLU,WR)
           write(sout,901)
        ENDIF  
        WRITE(SOUT,1) BANG,BRLU
        WRITE(SOUT,2) VANG,VRLU
        WRITE(SOUT,3) WA,WR
        RETURN
      ELSE
        WRITE(SOUT,*) 'Specify direction in reciprocal lattice.'
      ENDIF        
      END


       SUBROUTINE BRAG(ICOM)
C***********************************************************************
C if ICOM=0 ... TRAX matrix
C if ICOM=1 ... M.C. matrix
C***********************************************************************
       IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      
       REAL*8 EPS1
       PARAMETER(EPS1=1.D-6)
       INTEGER*4 I,J,ICOM
       REAL*8 A(4,4),HUITLOG2
       LOGICAL*4 LZER
       REAL*8 ZV1,ZV2,ZV3,DQX,DQY,DQZ,DEE,DVN,DUM
       REAL*8 GETFWHM
       
       
        IF (ICOM.EQ.1) THEN
          DO I=1,4
          DO J=1,4
             A(I,J)=ANESS(I,J)
          ENDDO
          ENDDO    
        ELSE
          DO I=1,4
          DO J=1,4
             A(I,J)=ATRAX(I,J)
          ENDDO
          ENDDO    
        ENDIF  
 

       HUITLOG2=8.D0*LOG(2.D0)
       LZER=.FALSE.
       DO 11 I=1,4
       IF (ABS(A(I,I)).LT.EPS1) LZER=.TRUE.
11       CONTINUE

       IF (LZER) THEN
       WRITE(SMES,*) ' BRAG: problem with matrix -> zeros on diagonal'
       RETURN
       ENDIF

C---------------------------------------------------------------------------

C   BRAGG WIDTHS IN RECIPROCRAL ANGSTROMS
       DQX=SQRT(HUITLOG2/A(1,1))
       DQY=SQRT(HUITLOG2/A(2,2))
       DQZ=SQRT(HUITLOG2/A(3,3))
C   ENERGY BRAGG AND VANADIUM WIDTHS.
       DEE=SQRT(HUITLOG2/A(4,4))
       CALL VANAD(A,DVN,DUM)

C   VANADIUM WIDTHS IN RECIPROCRAL ANGSTROMS       
       ZV1=GETFWHM(A,1)
       ZV2=GETFWHM(A,2)
       ZV3=GETFWHM(A,3)

       write(sout,5) DQX,DQY,DQZ,
     *                ZV1,ZV2,ZV3,
     *                CUNIT,DVN,DEE
5       FORMAT(' Bragg widths (radial,tangential,vertical) [A-1]'/    
     1       ' DQR=',F9.5,' DQT=',F9.5,' DQV=',F9.5,/,
     4  ' '/
     2  ' ''Vanad'' widths (from section at dE=0) [A-1]'/
     3       ' DQR=',F9.5,' DQT=',F9.5,' DQV=',F9.5,/,
     4  ' '/
     5       ' Energy widths (Vanad, Bragg) ',A,/,
     6       ' DVN=',F9.5,' DEE=',F9.5)
       RETURN
       END


C
C***********************************************************************
      SUBROUTINE RESOL(ICOM,IARG)
C Calculate resolutuion matrices and related parameters
C ICOM=1  ... analytically (TRAX)
C ICOM=2  ... Monte Carlo      
C***********************************************************************
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'

      REAL*8 ADA(4,4),B(4,4),AZ(4,4),A(4,4)
      REAL*8 TMP,VI,VF,VRES,R0PHON,R0BRAG
      INTEGER*4 ICOM,IARG,K,I,J
      CHARACTER*2 IAX(4),IAXP(4)
      CHARACTER*4 FWHM
      DATA IAX/'X','Y','Z','W'/,FWHM/'FWHM'/
      DATA IAXP/'X''','Y''','Z''','W'''/

900   FORMAT('Analytical result  (TRAX):')
901   FORMAT('Monte Carlo result (NESS):')
910   FORMAT('Vol(ki)= ',G12.4,'    Vol(kf)=',G12.4,
     *   '   Vol(QE)= ',G12.4)
911   FORMAT('Norm factors:   R0(C&N)= ',G12.4)
920   FORMAT('Resolution Matrix, Cooper & Nathans coordinates [A^-1]')
921   FORMAT(2x,4(10X,A))
922   FORMAT(2X,A,4E12.4)
930   FORMAT('Resolution Matrix, rec. lat. coordinates')
940   FORMAT('Diagonalised Resolution Matrix, [r.l.u.]')
941   FORMAT(2x,5(10X,A))
942   FORMAT(2X,A,5E12.4)
945   FORMAT('Direction Cosines (w.r.t. reciprocal lattice)')



      IF (ICOM.EQ.1) THEN
        DO I=1,4
        DO J=1,4
           A(I,J)=ATRAX(I,J)
        ENDDO 
        ENDDO 
        write(sout,900)  
      ELSE
        DO I=1,4
        DO J=1,4
           A(I,J)=ANESS(I,J)
        ENDDO   
        ENDDO 
        write(sout,901)  
      ENDIF
       
      IF (IARG.LE.0.OR.IARG.GT.4) THEN 
         GOTO 11
      ELSE  
         GO TO (11,12,13,14),IARG
      ENDIF   
      RETURN
C --------------------------------------------------------------------
C     RESOL 1, get volumes VI VF  etc. (J.S. 6/6/97)

11    call GETNORMS(ICOM,VI,VF,VRES,R0PHON,R0BRAG)
      write(sout,910) VI,VF,VRES
      write(sout,911) R0PHON   ! ,R0BRAG       
      RETURN

C --------------------------------------------------------------------
C     RESOL 2, resolution matrix in C&N coord.

12    write(sout,920)
      write(sout,921) IAX
      write(sout,922) (IAX(I),(A(I,J),J=1,4),I=1,4)
      RETURN
C --------------------------------------------------------------------
C     RESOL 3, resolution matrix in rec. lat. coord.

13    CALL CN2RLU(A,AZ)
      write(sout,930) 
      write(sout,921) IAX
      write(sout,922) (IAX(I),(AZ(J,I),J=1,4),I=1,4)      
      RETURN
C --------------------------------------------------------------------
C     RESOL 4, Diagonalized matrix

14    CALL CN2RLU(A,AZ)
      CALL DIAG(AZ,ADA,B)
      write(sout,940) 
      write(sout,921) IAXP 
      DO I=1,4
        DO  K=1,4
         IF (ADA(I,K).LT.1.D-20) ADA(I,K)=0.D0
        ENDDO
        WRITE(sout,922) IAXP(I),(ADA(I,K),K=1,4)
      ENDDO

C *** At Cooper&Nathans the res. function is exp(-.5*MijXiXj) !!!!!!
 
      TMP=8.D0*LOG(2.D0)
      DO I=1,4
         IF (ADA(I,I).GT.0.D0) ADA(I,I)= SQRT(TMP/ABS(ADA(I,I)))
      ENDDO
      write(sout,941) IAX,FWHM
      DO I=1,4
        WRITE(sout,942) IAXP(I),(B(J,I),J=1,4),ADA(I,I)
      ENDDO
      RETURN
        
      END

C***********************************************************************
C//
C//    D E P O S I T -  UNITS NOT USED IN CURRENT VERSION
C//
C***********************************************************************



C------------------------------------------------------------------------------------
      SUBROUTINE A3_TO_Q(A3,Q)
C Get QHKL equivalent to the sample rotation by A3 (nominal QHKL corresponds to A3=0)
C Uses current transformation matrix, MRC !
C------------------------------------------------------------------------------------ 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      REAL*8 A3,DA3,VQ(4),WQ(4),Q(3),SI,CO
      INTEGER*4 I

      IF(A3.NE.0.) THEN
        DA3=A3*deg
        SI=SIN(DA3)
        CO=COS(DA3)
        WQ(1)=Q0*(CO-1.D0)
        WQ(2)=-Q0*SI
        WQ(3)=0.
        WQ(4)=0.
        CALL MXV(1,4,4,MRC,WQ,VQ)   ! from CN to r.l.u.
        DO I=1,3
         Q(I)=RES_DAT(i_QH+I-1)+VQ(I)
        ENDDO
      ELSE
        DO I=1,3
         Q(I)=RES_DAT(i_QH+I-1)
        ENDDO        
      ENDIF
      END

C------------------------------------------------------------------
      SUBROUTINE PSDSCAN(QXMIN,QXMAX,QYMIN,QYMAX,EMIN,EMAX)
C     calculates direction and step size of the scan performed
C     by a linear position-sensitive detector
C------------------------------------------------------------------
      implicit NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'trax.inc'
      INCLUDE 'rescal.inc'

      INTEGER*4 I
      REAL*8 QXMIN,QXMAX,QYMIN,QYMAX,EMIN,EMAX
      REAL*8 TBR,COA,sigs,siga,SIA, SIN1,SIN2,RO,XDMAX,VKX,VKZ,HIR

      REAL*8 DQE(4),RQE(4)

1     FORMAT(2x,'PSD scan range (dQ,dE) : ',4(1x,F7.3),'  [meV]')
2     FORMAT(2x,'scale: ',F8.3,' mm/range')

      TBR=TETAA*TDR
      CALL HIRANG(TBR,HIANA,HIR)
      COA=(Q0**2-2*HOMEGA/HSQOVM)/(2*KF0*Q0)
      sigs=SIGN(1.D0,TETAS)
      siga=SIGN(1.D0,TETAA)
      SIA=SQRT(1-COA**2)


      SIN1=SIN(ABS(TBR-HIR))
      SIN2=SIN(ABS(TBR+HIR))
      IF(SIN1.LT.1.D-6) SIN1=1.D-6
      IF(SIN2.LT.1.D-6) SIN2=1.D-6
      RO=ROHA

      XDMAX=WANA*(SIN1 - 2.*RO*VL3 + VL3/VL2*SIN2)/2.

      VKX=-VKF*SIN2/VL2*WANA/2.

      VKZ=siga*VKF*(RO-SIN2/VL2)/ABS(TAN(TBR))*wana/2.

      QXMAX= VKZ*COA + VKX*SIA*sigs
      QYMAX=-VKZ*SIA*sigs + VKX*COA

      EMAX=-2*VKZ/KF0*EF0


      QXMIN=-QXMAX
      QYMIN=-QYMAX
      EMIN=-EMAX

      DQE(1)=(QXMAX-QXMIN)
      DQE(2)=(QYMAX-QYMIN)
      DQE(3)=0.
      DQE(4)=(EMAX-EMIN)

      CALL MXV(1,4,4,MRC,DQE,RQE)

      write(sout,*)
      write(sout,1) (RQE(I),I=1,4)
      write(sout,2) XDMAX*20.
      write(sout,*)

      RETURN
      END
