C////////////////////////////////////////////////////////////////////////
C $Id: restrax_main.f,v 1.13 2006/05/10 14:36:26 saroun Exp $
C
C  *** R E S T R A X ***
C written by J.Saroun and J.Kulda, Institute Laue-Langevin, Grenoble, 
C and Nuclear Physics Institute, Rez near Prague, November 1995-2005 
C 
C The program package for simulating the neutron optics of three-axis 
C spectrometers (TAS), optimization of their resolution and luminosity 
C and evaluation of experimental data collected with them. 
C The program code includes both a high-speed analytical (Gaussian) convolution 
C algorithm and a Monte Carlo ray-tracing method providing enhanced accuracy 
C in description of most of the spectrometer components.
C 
C Using in part: 
C 
C 1) RESCAL - provides all except of the graphics subroutines and 
C    the subroutine AFILL, which calculates the resolution matrix.  
C 
C    - written by M.  Hergreave & P.  Hullah, P.C.L.  London, July 1979 
C    - updated by P.  Frings, ILL Grenoble, October 1986 
C 
C 2) TRAX - provides a kernel replacing the subroutine AFILL and taking 
C    into account real dimensions of the spectrometer components and 
C    focusing by curved perfect and mosaic crystals.  
C 
C    - written by (*)M.Popovici, A.D.Stoica and I.Ionita, Institute for 
C    Nuclear Power Reactors, Pitesti, 1984-1986 
C    (*) University of Missouri
C    [1] M.Popovici, A.D.Stoica and I.Ionita, J.Appl.Cryst. 20 (1987),90.
C    [2] M.Popovici et al., Nucl.Instrum. Methods A338 (1994), 99.
C 
C 
C****************************************************************************
C *** For all additional information contact the authors:                 ***
C ***                                                                     ***
C ***                 kulda@ill.fr        saroun@ujf.cas.cz               ***
C ***                                                                     ***
C****************************************************************************
                         

C-------------------------------------------------------------
      SUBROUTINE BEFORE
C  Call whenever the setup may have changed
C  Updates calculated auxilliary fields and runs TRAX      
C------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'      
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      LOGICAL*4 LOG1
      INTEGER*4 IER
      COMMON /ERROR/IER
C ATTENTION : order of following routines is important !!!

      IER=0
c      write(*,*) 'BEFORE 1'
      CALL RECLAT          !   compute reciprocal lattice parameters and matrices
c      write(*,*) 'BEFORE 2'
      CALL SCATTRIANGLE    !   compute and check KI,KF,Q and tras. matrix Lab -> CN
c      write(*,*) 'BEFORE 3'
      CALL ANGSCAN(RES_DAT(i_DA3),0.D0) ! scan in DA3 => adjust DH,DK,DL and set DE=0
c      write(*,*) 'BEFORE 4'
      CALL TRANSMAT        !   create transformation matrices for coordinate systems
c      write(*,*) 'BEFORE 6'
      CALL TAS_TO_NESS     !   convert RESCAL+CFG to NESS
c     write(*,*) 'BEFORE 7'
      CALL TAS_TO_TRAX     !   convert RESCAL+NESS to TRAX      
c      write(*,*) 'BEFORE 8'
      CALL THRAX           !   calculate TRAX res. matrix             
c      write(*,*) 'BEFORE 9'
      CALL GetXSpec        !   prepare arrays for x-axis of current measured-spectrum
c      write(*,*) 'BEFORE 10'
c check whether  setup has been modified
      CALL SPEC_UNCHANGED(LOG1)
      isCHANGED=.NOT.LOG1
      CALL SPEC_GETCHK(checkSUM)
      CALL MFIT_GET(mf_cur)
      NEEDBEFORE=.FALSE.
c      write(*,*) 'BEFORE set ',mf_cur,' done: ',mf_done(mf_cur),
c     *           ' changed: ',isCHANGED
      END                                 

C-----------------------------
      SUBROUTINE LOGO
C----------------------------- 
      IMPLICIT NONE
      INCLUDE 'config.inc'
      INCLUDE 'const.inc'      
      INCLUDE 'inout.inc'

1     FORMAT(2x,'-----------------------------------------------',/,
     &       2x,'R E S T R A X  - Monte Carlo simulation of TAS ',/,
     &       2x,'Version: ',a40,/,
     &       2x,'Build:   ',a40,/,
     &       2x,'-----------------------------------------------',/,
     &       2x,'(C) J.Saroun & J.Kulda',/,
     &       2x,'ILL Grenoble, NPI Rez near Prague',/,
     &       2x,' ',/,
     &       2x,'Using (in part):',/,
     &       2x,'RESCAL by M.Hargreave, P.Hullah & P. Frings',/,
     &       2x,'TRAX   by M.Popovici, A.D.Stoica & I.Ionita',/,
     &       2x,'-----------------------------------------------',/,
     &       2x,'type ? for command list',/,/)
     

      write(sout,1) PACKAGE_VERSION,PACKAGE_DATE  
      return
      end

C-----------------------------------------------------------------------
      SUBROUTINE SETPATH(SARG)
C select search path for data files      
C-----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      
      CHARACTER*(*) SARG
      CHARACTER*128 PROMPT,ANSWER
      
      PROMPT=' Path to data files'
      ANSWER=' Data will be searched in'      
      CALL DLG_SETPATH(SARG,PROMPT,ANSWER,0,DATPATH)      
      END  
      
C-----------------------------------------------------------------------
      SUBROUTINE SETRESPATH(SARG)
C select search path for configuration files      
C-----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      
      CHARACTER*(*) SARG
      CHARACTER*128 PROMPT,ANSWER
      
      PROMPT=' Additional search path for configuration files'
      ANSWER=' Configurations will be searched in'      
      CALL DLG_SETPATH(SARG,PROMPT,ANSWER,0,RESPATH)
      END  

C--------------------------------
      SUBROUTINE DOSHELL(COMM)
c execute shell command      
c--------------------------------
      implicit none
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      CHARACTER*(*) COMM
      CHARACTER*256 COMM1
      INTEGER*4 IS,L
      
1     FORMAT(' Command : ',$)
2     format(A)
      COMM1=' '
      if((COMM(1:1).EQ.' ').OR.(COMM(2:2).EQ.CHAR(0))) THEN 
        write(SOUT,1)
        read(SINP,2) COMM1
      else
        L=LEN(COMM)
        IF(L.GT.256) L=256
        COMM1=COMM(1:L)
      endif
      CALL BOUNDS(COMM1,IS,L)
      
      IF (L.GT.0) THEN
        CALL SYSTEM(COMM1(IS:IS+L-1))
      ENDIF
      end
      
     
C-----------------------------------------------------
      SUBROUTINE LISTCFG
C  list available configuration files. 
C creates appropriate shell script and executes      
c-----------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INTEGER*4 IS,IL
      
1     FORMAT('#!/bin/sh',/,
     * 'for F in `ls ',a,'*.cfg` ; do',/, 
     * 'echo $F "        ... " `sed -n -e''2p'' $F`',/,
     * 'done',/)
            
      OPEN(FILE='~tmprestrax',UNIT=22,STATUS='UNKNOWN',ERR=99)
      CALL BOUNDS(RESPATH,IS,IL)
      WRITE(22,1,ERR=98) ' '
      IF (IL.GT.0) THEN
        WRITE(22,1,ERR=98) RESPATH(IS:IS+IL-1)      
      ENDIF
      CLOSE(22) 
      CALL DOSHELL('chmod 755 ~tmprestrax')
      CALL DOSHELL('./~tmprestrax')
      CALL DOSHELL('rm -f ./~tmprestrax')
      RETURN
      
      
98    CLOSE(22)      
99    RETURN

      END

C-------------------------------------------------------------
      SUBROUTINE SETCFG(SARG,IREAD)
C Read configuration file
C IF IREAD>0, prepare also all calculated fields and run TRAX!
c-------------------------------------------------------------
      IMPLICIT NONE
      CHARACTER*(*) SARG
      INTEGER*4 IREAD

      IF (IREAD.GT.0) THEN
         CALL TAS_READCFG(SARG)  ! read CFG
         CALL BEFORE       ! prepare all and run TRAX
      ELSE   
         CALL TAS_READCFG(SARG)  ! only read CFG
      ENDIF
      END


C-----------------------------------------------------------------------
      SUBROUTINE LIST
C-----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'restrax_cmd.inc'
      
      INTEGER*4 I1,I2,I
       
19     FORMAT(3(' ',A5,' = ',G14.7,1X))                     
51     FORMAT(' ',2(A4,' = ',F10.5,1X),/,                   ! DM,DA
     2        ' ',3(A4,' = ',F10.2,1X),/,                   ! ETAM,ETAA,ETAS
     3        ' ',3(A4,' = ',F10.0,1X),/,                   ! SM,SA,SS 
     4        ' ',A4,' = ',F10.5,1X, A4,' = ',F10.0,1X,/,  ! KFIX,FX 
     5        ' ',4(A4,' = ',F10.2,1X),/,                   ! ALF1..4 
     6        ' ',4(A4,' = ',F10.2,1X),/,                   ! BET1..4
     7        4(' ',3(A4,' = ',F10.4,1X)/),                 ! AS,AA,AX,BX     
     1        ' ',4(A4,' = ',F10.4,1X),/,                   ! QH..EN
     2        ' ',6(A4,' = ',F10.4,1X),/,                   ! DQH..DE,DA3,DA4
     3        ' ',4(A4,' = ',F10.4,1X),/,                   ! GH..GL,GMOD
     4        ' ',4(A4,' = ',F10.4,1X),/,                   ! ROMH..ROAV
     5        ' ',2(A4,' = ',F10.2,1X))                     ! SDI,SHI

       IF (NOS.GE.2)   THEN      
         I1=NINT(RET(1))
         I2=NINT(RET(2))
         write(sout,19)(RES_NAM(I),RES_DAT(I),I=I1,I2)
       ELSE
         write(sout,51) (RES_NAM(I),RES_DAT(I),I=1,RES_NVAR)   
         nos=0
         call SET_3AX(1)
         call SET_3AX(2)
         call SET_3AX(3)
         call SET_3AX(4)
         call SET_3AX(5)
         call SET_3AX(6)
       ENDIF
       END
       
C***********************************************************************
        SUBROUTINE UNITS(SARG)
C***********************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      CHARACTER*(*) SARG

2     FORMAT('[THz]')
5     FORMAT('[meV]')
4     FORMAT(' Units are ',A)
        
      IF(SARG(1:1).EQ.'T') THEN
          EUNI=0.24181
          WRITE(CUNIT,2)
      ELSE
          EUNI=1.
          WRITE(CUNIT,5)
      ENDIF
      write(sout,4) CUNIT
      END
      
C-----------------------------------------------------------        
      SUBROUTINE GETROANAL(RO)
C generates "optimal" monochromator and analyzer curvatures
C using analytical expression:
C - focus parallel beam at the sample by monochromator 
C - set monochromatic focusing on analyzer    
C-----------------------------------------------------------      
              
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'trax.inc'

      REAL*4 RO(4)
      INTEGER*4 IERR
      REAL*8 THA,CHIA,THM,CHIM
      REAL*8 R(RES_NVAR)
      COMMON /ERROR/IERR
      EQUIVALENCE(R(1),RES_DAT(1))
                           
      IF (R(i_SM).NE.0) THEN
         THM=PI/SQRT(EI0/HSQOV2M)/R(i_DM)
         THM=ABS(ASIN(THM))
         CHIM=-HIMON*TDR
         RO(1)=SIN(THM+CHIM)/VL1*100
         RO(2)=1./VL1/(2.*SIN(THM)*COS(CHIM))*100
      ENDIF       
      IF (R(i_SA).NE.0) THEN
         THA=PI/SQRT(EF0/HSQOV2M)/R(i_DA)
         THA=ABS(ASIN(THA))
         CHIA=-HIANA*TDR
c         RO(3)=(VL2*SIN(THA+CHIA) + VL3*SIN(THA-CHIA))/2./VL2/VL3*100
         RO(3)=SIN(THA-CHIA)/VL2*100
         RO(4)=(1./VL2+1./VL3)/(2.*SIN(THA)*COS(CHIA))*100
      ENDIF       
      END
          
C-----------------------------------------------------------        
      SUBROUTINE GETRO(ICOM)
C generates "optimal" monochromator and analyzer curvatures 
C ICOM=0 ... focus parallel beam at the sample and from the 
C            sample to the detector
C *** modified J.S. 3/6/1997:  
C ICOM=1 ... optimize Vanad resolution and intensity
C *** modified J.S. 10/12/2002:     
C ICOM=0 ... optimize for Rowland focusing on analyzer
C-----------------------------------------------------------      
              
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'

      CHARACTER*10 REMARK,VNAME(4)
      REAL*4 RO(4),ROMI(4),ROMA(4)
      REAL*4 OPTV,TOL
      INTEGER*4 ICOM,I,IERR
      REAL*8 R(RES_NVAR)
      COMMON /ERROR/IERR
      EQUIVALENCE(R(1),RES_DAT(1))
      EXTERNAL OPTV     
      DATA VNAME /' ROMH = ',' ROMV = ',' ROAH = ',' ROAV = '/
      DATA ROMI /-10.0,-10.0,-10.0,-10.0/
      DATA ROMA /10.0,10.0,10.0,10.0/
           
1     FORMAT(a,F8.4,A20)
5     FORMAT(' Numerical optimization failed ! ') 
      
      
      CALL GETROANAL(RO)          
      TOL=0.001
      IF (R(i_SA)*R(i_SM).NE.0.AND.ICOM.EQ.1) THEN
        CALL LMOPT(OPTV,RO,ROMI,ROMA,4,TOL,1)
        write(SOUT,*)        
      ENDIF  
      IF (IERR.LT.0) WRITE(SMES,5)
      
      DO I=1,4                  
        IF((NOS.EQ.0).OR.((NOS.GE.I).AND.(RET(I).EQ.1.))) THEN
           RES_DAT(I+I_ROMH-1)=RO(I)
           REMARK=' changed'
        ELSE
           REMARK=' '   
        ENDIF
        WRITE(SOUT,1) VNAME(I),RO(I),' [m-1]  '//REMARK   
      ENDDO  
      
      DO I=I_ROMH,I_ROAV
         mf_par(I,mf_cur)=RES_DAT(I)
      ENDDO   
                
      
      RETURN
      END
                
C--------------------------------------------------------------        
      SUBROUTINE OPTINSTR
C Optimise TAS with respect to the IPAR-th parameter
C IFPAR ... parameter from EXCI model used as a figure of merit
C 
C--------------------------------------------------------------                    
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'res_grf.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'
      INCLUDE 'exciimp.inc'
            
      RECORD /MODEL/ rm

      CHARACTER*128 LINE
      REAL*4 A,A0,AMI,AMA
      REAL*8 LIM(2)
      REAL*4 OPTTAS,TOL
      INTEGER*4 IPAR,IFPAR,GETICOM,IERR
      COMMON /ERROR/IERR
      EXTERNAL OPTTAS     
           
1     FORMAT(a)
2     FORMAT(a,' = ',F10.4)
      
      
      CALL getmodel(rm)
      CALL GENDT
            
20    CALL DLG_STRING('TAS parameter to be optimised (name)',LINE,0)      
      IPAR=GETICOM(line)
      IF (IPAR.LE.0.OR.IPAR.GT.RES_NVAR) GOTO 20
      write(sout,1) RES_NAM(IPAR)//' will be optimised'       
      
30    CALL DLG_INPUT('lower limit:upper limit',LIM,0)
      IF (LIM(1).GE.LIM(2)) GOTO 30
      AMI=LIM(1)
      AMA=LIM(2)
      
      CALL DLG_INTEGER('EXCI parameter as figure of merit',IFPAR,0,3,rm.NTERM)
      write(sout,1) rm.PARNAME(IFPAR)//' is the figure of merit'
      
      SILENT=2
      OPTMERIT=1
      OPTEV=10000
      TOL=0.1
      IF (NOS.GT.0) THEN
        OPTEV=NINT(1000*RET(1)) ! number of events
      ENDIF
C// parameter 2 is used by GENDT as the integral for simulated data
      IF (NOS.GT.2) THEN
        OPTMERIT=NINT(RET(3))  ! figure of merit (see OPTTAS)
      ENDIF
      IF (NOS.GT.3) THEN
        TOL=RET(4)             ! TOL
      ENDIF
      write(sout,*) 'FM=',OPTMERIT                
      OPTFPAR=IFPAR
      OPTPAR=IPAR
      OPTDPAR=FPAR(IFPAR)*0.01
      A=RES_DAT(OPTPAR)
      A0=A
      CALL LMOPT(OPTTAS,A,AMI,AMA,1,TOL,0)  
      SELECT CASE (IERR)
        CASE (1) 
           WRITE(SOUT,*) 'OK, exact match' 
        CASE (2) 
           WRITE(SOUT,*) 'OK, exit on TOL limit' 
        CASE (-1)
           WRITE(SOUT,*) 'not finished, stalled on lambda' 
        CASE (-2)
           WRITE(SOUT,*) 'not finished, iteration number limit' 
      END SELECT                        
      WRITE(SOUT,2) RES_NAM(OPTPAR),A 
      SILENT=DSILENT ! return to default
      GRFARG(0)=4
      call PLOTOUT   ! plot the result
      END     



***********************************************************************
      SUBROUTINE REINP(SARG)
C     redirection of input
C***********************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      character*(*) SARG
      CHARACTER*128 INPFILE
      INTEGER*4 IUINI,IUEXT,IRES
      DATA IUINI,IUEXT/0,10/
      
      IF(IUINI.EQ.0) IUINI=SINP      
      IF (SINP.NE.IUINI) CLOSE(SINP)
      if((SARG(1:1).ne.' ').and.(SARG(1:1).ne.char(0))) THEN
          CALL OPENRESFILE(SARG,'inp',IUEXT,0,0,INPFILE,IRES) 
          IF (IRES.NE.0) GOTO 2002        
          SINP=IUEXT
      else
          SINP=IUINI
      endif   
      RETURN     

2002  write(smes,*) 'Cannot open input file '//SARG
      SINP=IUINI
      END 

***********************************************************************
      SUBROUTINE REOUT(SARG)
C     redirection of input
C***********************************************************************
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      character*(*) SARG
      INTEGER*4 IUINI,IUEXT
      DATA IUINI,IUEXT/0,11/
      
      IF(IUINI.EQ.0) IUINI=SOUT      
      IF (SOUT.NE.IUINI) CLOSE(SOUT)
      if((SARG(1:1).ne.' ').and.(SARG(1:1).ne.char(0))) THEN
          OPEN (UNIT=IUEXT,ERR=2002,FILE=SARG, STATUS='UNKNOWN')          
          SOUT=IUEXT
      else
          SOUT=IUINI
      endif   
C      WRITE(*,*) SOUT
      RETURN     

2002  write(smes,*) 'Cannot open output file  '//SARG
      SOUT=IUINI
      END 
      
***********************************************************************
      SUBROUTINE EMODE
C     Switch on/off elastic mode (for SA=0 only)
C !! J.S. 5/4/2002, allow EMOD also for SA<>0
C***********************************************************************
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax_cmd.inc'

      LOGICAL*4 EMOD
      COMMON /MODE/ EMOD
      DATA EMOD /.FALSE./
      
      IF(NOS.EQ.0) THEN
         IF(EMOD) THEN
            write(sout,*) 'Elastic scattering'
         ELSE
            write(sout,*) 'Inelastic scattering'
         ENDIF
      ELSE IF (RET(1).EQ.1.) THEN
c         IF(SA.EQ.0) THEN
            EMOD=.TRUE.
            write(sout,*) 'Elastic scattering mode is On'   
c         ELSE
c            EMOD=.FALSE.
c            write(sout,*) 'Elastic scattering mode is possible only'//
c     *  ' with SA=0 !'
c         ENDIF
      ELSE IF (RET(1).EQ.0.) THEN
            EMOD=.FALSE.
            write(sout,*) 'Elastic scattering mode is Off' 
      ENDIF        
                    
      END          
      


***********************************************************************
      LOGICAL*4 FUNCTION CMDFILTER(ICMD)
C  filter for commands for special modes. Set .false. if the command is forbidden
C***********************************************************************
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'

      INTEGER*4 ICMD,SM,SA
      LOGICAL*4 LOG,EMOD
      COMMON /MODE/ EMOD
      
      SM=RES_DAT(i_SM)
      SA=RES_DAT(i_SA)
      
      LOG=.TRUE.
      IF (EMOD.OR.(SA*SM.EQ.0)) THEN   ! forbid for elastic and SA,SM=0 modes  
c// forbid all commands using TRAX
        IF (RES_NAM(ICMD).EQ.'BRAG') LOG=.FALSE.     ! BRAG
        IF (RES_NAM(ICMD).EQ.'PHON') LOG=.FALSE.     ! PHON        
        IF (RES_NAM(ICMD).EQ.'RES')  LOG=.FALSE.     ! RES
        IF (RES_NAM(ICMD).EQ.'FIT')  LOG=.FALSE.     ! FIT
        IF (SA*SM.EQ.0.AND.(.NOT.LOG)) THEN
            write(SOUT,*) 'SM or SA = 0 => Only Monte Carlo is accepted'
        ENDIF       
        IF (EMOD.AND.(.NOT.LOG)) THEN
            write(SOUT,*) 'Elastic mode => Only Monte Carlo is accepted'
        ENDIF       
      ELSE
        CMDFILTER=.TRUE.
      ENDIF
      
      END
           
***********************************************************************
      SUBROUTINE MAKEMC(SARG)
C calls Monte Carlo if required 
C also set the SWRAYTR switch 
C***********************************************************************
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'res_grf.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'

      INTEGER*4 N,NMIN,IS,IL
      LOGICAL*4 LOG1,EMOD,LASTMODE
      CHARACTER*(*) SARG
      CHARACTER*128 S
      COMMON /MODE/ EMOD
      DATA LASTMODE /.FALSE./       
      SAVE LASTMODE
      IS=1
      CALL FINDPAR(SARG,1,IS,IL)
      S=SARG(IS:IS+IL-1)
      LOG1=.FALSE.

C// Set LOG=.TRUE. if the command requires MC simulation

C// commands with 0 arguments
      NMIN=0 
      IF (S(1:IL).EQ.'MBRAG') LOG1=.TRUE.           
      IF (S(1:IL).EQ.'MPHON') LOG1=.TRUE.           
      IF (S(1:IL).EQ.'MFIT')  LOG1=.TRUE.          
      IF (S(1:IL).EQ.'OPTAS') LOG1=.TRUE.          
      IF (LOG1) GOTO 10
      
C// commands with 1 arguments
      NMIN=1
      IF (S(1:IL).EQ.'PLOT') THEN  
         LOG1=(
     1    (GRFARG(0).EQ.-5).OR.
     1    (GRFARG(0).EQ.-4).OR.
     1    (GRFARG(0).EQ.-3).OR.
     1    (GRFARG(0).EQ.2).OR.
     2    (GRFARG(0).EQ.3).OR.
     3    (GRFARG(0).EQ.4.AND.iand(WHATHIS,2).EQ.2).OR.
     4    (GRFARG(0).EQ.5.AND.iand(WHATHIS,2).EQ.2).OR.
     5    (GRFARG(0).EQ.9).OR.
     5    (GRFARG(0).EQ.15).OR.
     5    (GRFARG(0).EQ.16).OR.
     & .FALSE.)
      ENDIF  
      IF (S(1:IL).EQ.'MRES') LOG1=.TRUE. 
      IF (LOG1) GOTO 10
      
C// commands with 3 arguments
      NMIN=3   
      IF (S(1:IL).EQ.'MFWHM')  LOG1=.TRUE.

C// execute ray-tracing
10    IF (LOG1) THEN
         SWRAYTR=1 ! switch ray-tracing on
         IF (NOS.GT.NMIN) THEN
            N=NINT(RET(NOS)*1000)
         ELSE   
            N=LASTNEV
         ENDIF 
         IL=0  
         IF(LASTMODE.XOR.EMOD) IL=1 ! mode has changed, run MonteCarlo anyway
         CALL RUNMC(IL,N)  ! call MonteCarlo 
         LASTMODE=EMOD
c         write(*,*) 'MAKEMC: ',S(1:IL),SWRAYTR 
      ELSE  
         SWRAYTR=0 ! switch ray-tracing off      
c         write(*,*) 'MAKEMC: ',S(1:IL),SWRAYTR 
      ENDIF  
       
      END
 

C-----------------------------------------------------------        
      SUBROUTINE MCPHON
C Wrapper for MCPROFIL: scans through planar dispersion (GH..GMOD)
C Returns phonon widths in gaussian approximation
C Plots PAGE 2
C ICOM=1 .. TRAX
C ICOM=2 .. Ray tracing
C-----------------------------------------------------------      
      IMPLICIT NONE 

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'res_grf.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      REAL*8 Z,GETSQOM
      REAL*4 eval,HIST_LIN
      REAL*8 PHON_PO,PHON_POR,PHON_POS,PHON_MC,PHON_MCR,PHON_MCS
      COMMON /PHON/ PHON_PO,PHON_POR,PHON_POS,PHON_MC,PHON_MCR,PHON_MCS
         
C Calculate and show scan profile
      Z=GETSQOM(1,mf_max) ! fill QOM array with events        
      CALL HISTINIT  ! initialize histogram
      eval = HIST_LIN(XHIST,RHIST,NHIST,nhist(MDAT),mf_cur) ! generate histogram using planar dispersion

C Print phonon peak parameters

      if(SWRAYTR.EQ.1) then
        CALL GETPHONWIDTH(ANESS,PHON_MC,PHON_MCR,PHON_MCS) 
          write(sout,*) 'Phonon FWHM, ray-tracing '//
     1'resolution function:'      
          write(sout,906) PHON_MC,CUNIT,PHON_MCR,CUNIT,PHON_MCS
          write(sout,907) HNORM(mf_cur)                 ! norm from M.C.
      else 
        CALL GETPHONWIDTH(ATRAX,PHON_PO,PHON_POR,PHON_POS) 
          write(sout,*) 'Phonon FWHM, analytical (gaussian) '//
     1'resolution function:'      
          write(sout,906) PHON_PO,CUNIT,PHON_POR,CUNIT,PHON_POS    
          write(sout,907) HNORM(mf_cur)                 ! norm from TRAX
      endif

906   FORMAT(F8.4,' [A-1]',A5,' ',F8.4,' [r.l.u.]'
     1         ,A5,' ',F8.4,' [steps]')
907   FORMAT(1X,'Integrated intensity:',G11.5)
        
      GRFARG(0)=4
      CALL PLOTOUT      

      END 
C

C-----------------------------------------------------------        
      SUBROUTINE GENDT
C simulation of scan data incl. counting errors      
C-----------------------------------------------------------      
      IMPLICIT NONE 
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'res_grf.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'
      
      INTEGER*4 NE
      REAL*8 SUMA
      DATA SUMA/1.D4/ ! set default and remember

      SWRAYTR=1 ! ray-tracing on
      NE=10000  ! default number of events
      IF (NOS.GE.1) NE=INT(RET(1)*1000) ! number of events from the 1st argument
      IF (NOS.GE.2) SUMA=RET(2)    ! set optionally the sum of counts as the 2nd argument    
      CALL mfit_set(1)
      CALL DELDATA(1,mf_max)    ! clear channels
      CALL ADDDATA('channel',1,mf_cur,2)  ! add a dataset
      CALL SIMDATA(SUMA,NE,1)   ! simulate data incl. errors
      GRFARG(0)=4
      call PLOTOUT   ! plot the result
c      CALL WriteHist(' ')       ! save results
      END

C----------------------------------------------------------------------------- 
      SUBROUTINE EXPORT_RES(ARG)
C Exports resolution function to an ASCII file or std. output (SOUT)
C J. Saroun, 19/5/2000 
C----------------------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'restrax.inc'
      character*(*) ARG
      INTEGER*4 IU,N,I,J,L
      REAL*8 QE(4),P
     
1     FORMAT('Qx',12x,'Qy',12x,'Qz',12x,'dE',12x,'P')
2     FORMAT(5(G12.5,2x))        
3     FORMAT(I8,3x,3(G12.6,2x),'    ... events, (KI,Q,E)' )



      call KSTACK_N(N,mf_cur)
      IF(N.GT.0) THEN
        if((ARG(1:1).ne.' ').and.(ARG(1:1).ne.char(0))) THEN
          OPEN (UNIT=24,ERR=2002,FILE=ARG, STATUS='UNKNOWN')
          IU=24          
        else
          IU=SOUT
        endif
        write(IU,3) N,STP.KI,STP.Q,STP.E
        write(IU,1)
        DO I=1,N 
          CALL GETQE(I,mf_cur,QE,P)
          write(IU,2) (QE(J),J=1,4),P
        ENDDO
        IF(IU.EQ.24) CLOSE(IU)
      ENDIF
      RETURN

2002  L=LEN(ARG)
      write(SMES,*) 'Cannot open resol. function, file ',ARG(1:L)    
      return
      END

C----------------------------------------------------------------------------- 
      SUBROUTINE IMPORT_RES(ARG)
C Imports resolution function from an ASCII file or std. input (SINP)
C J. Saroun, 19/5/2000 
C----------------------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'restrax.inc'
      character*(*) ARG
      INTEGER*4 IU,N,I,J,L
      REAL*8 QE(4),P,KI,QQ,E0,EPS
      PARAMETER (EPS=1E-5)
     
1     FORMAT('WARNING: R(Q,E) for diferent scattering triangle !')

      if((ARG(1:1).ne.' ').and.(ARG(1:1).ne.char(0))) THEN
          OPEN (UNIT=24,ERR=2002,FILE=ARG, STATUS='OLD')          
          IU=24          
      else
          IU=SINP
      endif
      read(IU,*,ERR=2001) N, KI,QQ,E0
      IF((ABS(KI-STP.KI).GT.EPS*STP.KI).OR.
     *   (ABS(QQ-STP.Q).GT.EPS*STP.Q).OR.
     *   (ABS(E0-STP.E).GT.EPS*ABS(STP.E))) write(smes,1) 
      
      read(IU,*,ERR=2001)
      CALL KSTACK_ALLOCATE(N,mf_cur)
      DO I=1,N
          READ(IU,*,ERR=2001)(QE(J),J=1,4),P  
          CALL SETQE(I,mf_cur,QE,P)
      ENDDO
      IF(IU.EQ.24) CLOSE(IU)
      CALL GETCOV_QE(N)
      CALL BEFORE
      CALL mfit_get(mf_cur)

      RETURN


2001  write(SMES,*) 'Format error while loading resol. function '    
      return

2002  L=LEN(ARG)
      write(SMES,*) 'Cannot open resol. function, file ',ARG(1:L)    
      return

      END
      
C----------------------------------------------------------------------
      SUBROUTINE PRINTOUT
C Prints text with resolution parameters etc.
C----------------------------------------------------------------------
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'config.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'res_grf.inc'
      INCLUDE 'restrax.inc'

      INTEGER*4 STMP,IUSER,INOW,I
      CHARACTER*20 USER,NOW

500   FORMAT(1X,70('-'))
1     FORMAT(' ***  R E S T R A X  ',A,' ***')
2     FORMAT(' configuration: ',A60)
3     FORMAT(' user:          ',A60)
4     FORMAT(' results have been saved in restrax.txt')
      CALL PGQINF('USER',USER,IUSER)
      CALL PGQINF('NOW',NOW,INOW)

      OPEN(UNIT=20,FILE='restrax.txt',STATUS='UNKNOWN')
      WRITE(20,1) trim(PACKAGE_VERSION)
      WRITE(20,*)
      WRITE(20,2) CFGNAME
      WRITE(20,*) 'user:          '//USER(1:IUSER)//'    '//NOW(1:INOW)
      WRITE(20,500)

      STMP=SOUT
      SOUT=20
      I=SWPLOT
      SWPLOT=0
      CALL BRAG(1)
      WRITE(20,500)
      CALL BRAG(2)
      WRITE(20,500)
      CALL MCPHON(1)
      WRITE(20,500)
      CALL MCPHON(2)
      WRITE(20,500)
      CALL RESOL(1,1)
      WRITE(20,500)
      CALL RESOL(1,2)
      WRITE(20,500)
      CALL RESOL(1,3)
      WRITE(20,500)
      CALL RESOL(1,4)
      WRITE(20,500)
      CALL RESOL(2,4)
      WRITE(20,500)
      CLOSE(20)
  
      SOUT=STMP

      CALL PRINTFILE('restrax.txt')
      SWPLOT=I
      WRITE(SOUT,4)
      END


C----------------------------------------------------------------------
      SUBROUTINE PRINTFILE(SARG)      
C Print file named SARG
C try commands in following order:
C 1) $PGPLOT_ILL_PRINT_CMD sarg
C 2) $PRINTER sarg
C 3) lpr sarg
C----------------------------------------------------------------------
      IMPLICIT NONE
      CHARACTER*(*) SARG
      CHARACTER*40 PRN_COMMAND
      INTEGER*4 IS,IL
      CALL BOUNDS(SARG,IS,IL)
      IF (IL.GT.0) THEN
         PRN_COMMAND=' '
         CALL GETENV('PGPLOT_ILL_PRINT_CMD',PRN_COMMAND)
         IF (PRN_COMMAND.EQ.' ') CALL GETENV('PRINTER',PRN_COMMAND)         
         IF (PRN_COMMAND.EQ.' ') PRN_COMMAND='lpr' 
         CALL SYSTEM(PRN_COMMAND//' '//SARG(IS:IS+IL-1))
      ENDIF
      END

C*****************************************************************************
      SUBROUTINE REPORTOMEXC
C Print excitation energies and S(Q,E) from the EXCI module.
C Input [h k l] is either command option or QH .. QL for current dataset
C*****************************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'
      INCLUDE 'exciimp.inc'
            
      RECORD /MODEL/ rm
      
      INTEGER*4 I,INITDONE     
      REAL*8 DUM4(4),DUM6(6),dum61(6)
      DATA INITDONE/0/
      
6     format('OMEXC at Q=(',3(1x,G9.3),') E=',G9.3,':')
7     format('E[meV] : ',6(1x,G11.4))
8     format('S(Q,E) : ',6(1x,G11.4))

c      IF (INITDONE.EQ.0) THEN
        CALL INITEXCI(0,0)
c        INITDONE=1
c      ENDIF 
      IF (NOS.GE.3) THEN
        DO I=1,3
          DUM4(I)=RET(I)     
        ENDDO
      ELSE
        DO I=1,3
          DUM4(I)=mf_par(i_QH+I-1,mf_cur)
        ENDDO                 
      ENDIF
      IF (NOS.GE.4) THEN
        DUM4(4)=RET(4)
      ELSE
        DUM4(4)=mf_par(i_EN,mf_cur)
      ENDIF
      CALL EXCI(-2,DUM4,DUM6,DUM61)
      CALL getmodel(rm)
      write(sout,6) (DUM4(i),i=1,4)
      write(sout,7) (DUM6(i),i=1,rm.nbr)
      write(sout,8) (DUM61(i),i=1,rm.nbr)
      END


C-------------------------------------------
      SUBROUTINE FCONE_INI
C Set the flat-cone analyzer mode on/off
C depending on command argument RET(1)=1/0
C Creates analyzer channels as multiple datasets
C Dialog arguments:
C DLGARG(1) ... number of channels
C-------------------------------------------
      IMPLICIT NONE 
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'
      
      CHARACTER*2 CHN
      INTEGER*4 I,NP 
      DATA NP/32/ ! default number of channels
      
2     FORMAT(I2)
3     FORMAT('DA4=0, zero step between channels !',//,
     &       'Set DA4<>0 and repeat the command')

7     FORMAT(' Analyzer part is in normal position')
71    FORMAT(' Analyzer part is turned up')
72    FORMAT(' Analyzer part is turned down')
73    FORMAT(' Analyzer part is turned up/down')


      IF(NOS.GE.1) THEN
         IF (RET(1).EQ.1) THEN
             CFGMODE=1
         ELSE
             CFGMODE=0
         ENDIF
         mf_CFGMODE(mf_cur)=CFGMODE
      ENDIF
      IF (CFGMODE.EQ.0) WRITE(SOUT, 7)
      IF (CFGMODE.EQ.1.AND.STP.SA.GT.0) WRITE(SOUT,71) 
      IF (CFGMODE.EQ.1.AND.STP.SA.LT.0) WRITE(SOUT,72)
      IF (CFGMODE.EQ.1.AND.STP.SA.EQ.0) WRITE(SOUT,73)
      IF (NOS.LE.0) RETURN ! only report state
      
      IF (CFGMODE.EQ.1) THEN
c get number of channels
        IF (CMDMODE.EQ.1) THEN ! interactive mode
          CALL DLG_INTEGER('number of channels',NP,1,1,MDAT)
        ELSE ! take NP from the argument array
          NP=NINT(DLGARG(1))
          IF (NP.GT.MDAT) NP=MDAT
          IF (NP.LT.1) NP=1
        ENDIF
c check DA4<>0, otherwise switch to normal mode and exit        
        IF (NP.GT.1.AND.RES_DAT(i_DA4).EQ.0.D0) THEN
          write(smes,3)
          CFGMODE=0
          mf_CFGMODE(mf_cur)=CFGMODE
          RETURN
        ENDIF

c clear channels
        CALL mfit_set(1)
        CALL DELDATA(1,mf_max)

c uses QHKL from the 1st channel as the starting point
        CALL ADDDATA('channel01',1,mf_cur,2)
        CALL BEFORE

c scan through all other channels
        DO I=2,NP
          WRITE(CHN,2) I
          IF(CHN(1:1).EQ.' ') CHN(1:1)='0'
          CALL ROTA4(RES_DAT(i_DA4)*deg,RES_DAT(i_QH)) ! add setp in A4
          CALL ADDDATA('channel'//CHN,1,mf_cur+1,2)
          CALL BEFORE
        ENDDO   
        CALL mfit_set(1) ! return to the 1st channel
      ENDIF
      END

C----------------------------------------------- 
      SUBROUTINE RUNMC(COND,NEV)
c Run MC for all loaded data with NEV events
C All calls of Monte Caro should be made through this subroutine !!
c COND=1 ... run anyway
C COND=0 ... run only when necessary
C if NEV<=0 => prompt for number of events     
C Dialog arguments:
C DLGARG(0) ... number of events / 1000
C-----------------------------------------------       
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'
      
      INTEGER*4 COND,NEV,I,cur0,N
      DATA N/0/
      
      cur0=mf_cur
      DO I=1,mf_max  
        CALL mfit_set(I)
        IF (NEV.GT.0) N=NEV
        IF (NEV.LE.0.OR.N.EQ.0.OR.N.GT.MAXNEV) THEN  
          IF (CMDMODE.EQ.1) THEN
            CALL DLG_DOUBLE('Number of events / 1000',DLGARG(0),0,1.D-2,2.D+2)
          ENDIF
          N=INT(DLGARG(0)*1000)
        ENDIF  
	IF (N.GT.0) CALL IFNESS(COND,N)
      ENDDO
      if (mf_cur.ne.cur0) CALL mfit_set(cur0)
      END  

C-----------------------------------------------------
      SUBROUTINE READINIFILE(JOBNAME)
C   read initialization file
C CFGNAME = configuration file  
C DATAPATH = path to the data files   
C OPENFILE = data or RESCAL file to open   
C return JOBNAME .. filename of a job file to be executed at the startup 
c-----------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'restrax.inc'
      CHARACTER*128 LINE,S
      CHARACTER*(*) JOBNAME
      CHARACTER*128 INIFILE
      INTEGER*4 IRES,IS,IL,IERR
      LOGICAL*4 LOG1
      REAL*8 Z
      
1     FORMAT(A)
      JOBNAME=' '
      CALL OPENRESFILE('restrax.ini','ini',22,0,0,INIFILE,IRES)
      IF(IRES.EQ.0) THEN
         DO WHILE(IRES.EQ.0)
           READ(22,1,END=100,iostat=IRES) LINE
           IF(LINE(1:1).NE.'#') THEN 
C start with this configuration file: 
             CALL READ_STR('CFGNAME',LINE,CFGNAME,IERR)
C search data files in this directory
             CALL READ_STR('DATAPATH',LINE,DATPATH,IERR)
C Set default EXCI module (overrides command-line option)
             CALL READ_STR('EXCI',LINE,EXCILIB,IERR)             
C Set startup jobfile
             CALL READ_STR('JOB',LINE,JOBNAME,IERR)
C Set fit tolerance
             CALL READ_R8('FITTOL',LINE,Z,IERR) 
             IF (IERR.GE.0) FITTOL=Z
C Set initial weight on steepest descent  
             CALL READ_R8('FITLAM0',LINE,Z,IERR) 
             IF (IERR.GE.0) FITLAM0=Z

C Open this file first
             CALL READ_STR('OPENFILE',LINE,S,IERR)
C/ for OPENFILE, test whether it is data file or a *.res file
             IF (IERR.GE.0) THEN
               IS=1
               CALL FINDPAR(S,1,IS,IL)
               LOG1=(IL.GT.4) 
               IF (LOG1) LOG1=(LOG1.AND.S(IS+IL-4:IS+IL-1).EQ.'.res') ! stupid construct, required by Absoft debugger ...              
               IF (LOG1) THEN ! RESCAL file
                 DATNAME=' '
                 RESCAL_NAME=S(IS:IS+IL-1)
               ELSE                             ! data file
                 DATNAME=S(IS:IS+IL-1)
                 RESCAL_NAME=' '
               ENDIF  
             ENDIF   
           ENDIF  
         ENDDO
100      CLOSE(22)      
      ENDIF
      END
       


C----------------------------------------------------------------------------- 
      SUBROUTINE RESINIT
C// initialize RESTRAX
C// include all actions necessary to allocate memory, 
C// initialize variables, print LOGO etc..                
C----------------------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'res_grf.inc'
      INCLUDE 'config.inc'
      INCLUDE 'restrax.inc'
      INCLUDE 'restrax_cmd.inc'

      INTEGER*4 I,M,J,IL,IS,IRES
      CHARACTER*60 S1
      REAL*4 GASDEV1

      CHARACTER*10 prompt
      character*128 S,JOBNAME
      INTEGER*4 ISEED,IRND
      COMMON /RNDGEN/ ISEED,IRND
      INTEGER*4 IARGC

      CALL MKUPCASE(SYSNAME)
      IF (SYSNAME(1:7).EQ.'WINDOWS') THEN
        PATHDEL='\'
      ENDIF
c      REAL*8 DUM4(4),DUM6(6),dum61(6)
            
C normalizing constant for intensities
      ZNORM=2.D+6*HSQOV2M*SQRT(24/PI)   ! monitor efficiency included      

C initialize LINP
      prompt='ResTrax'
      CALL LINPSET(RES_NVAR+RES_NCMD,prompt,RES_NAM,RES_HLP)
      CALL LINPSETIO(SINP,SOUT,SMES)

C Handle command-line options
      M=IARGC()
      
C Derive path to default setup files from executable pathname
      
      call GETARG(0,S)
      i=INDEX(S,PATHDEL//'bin'//PATHDEL)       
      IF (i.gt.1) then
         CFGPATH=S(1:i)//'setup'//PATHDEL
      ELSE
         CFGPATH='setup'//PATHDEL
      ENDIF  
 
 
C     I=IGETARG(1,S,30)
      DO I=1,M
         call GETARG(I,S)
         if (S(1:2).eq.'-s') THEN
            read(S(3:30),*) J
            if(J.NE.0) THEN 
               ISEED=ABS(J) 
               write(SOUT,*) 'SEED=',ISEED
            ENDIF
         endif
         if (S(1:3).eq.'-gs') THEN
            read(S(4:30),*,err=9) GRFSAVE 
            write(SOUT,*) 'GRFSAVE=',GRFSAVE 
            GOTO 10             
9           GRFSAVE=0
         endif
10       if (S(1:2).eq.'-t') THEN
            read(S(3:30),*) J
            CALL RAN1SEED(ISEED)
            write(SOUT,*) 'Test of the random number generator:'
            call RAN1TEST(J,1000000*J)
            GOEND=1
         endif
         if (S(1:4).eq.'-sil') THEN
            read(S(5:30),*,err=12) DSILENT 
            write(SOUT,*) 'SILENT=',DSILENT 
         endif
12       if (S(1:5).eq.'-ran1') THEN
            IRND=1
            write(SOUT,*) 'Numerical Recipes RAN1 generator'
         endif
         if (S(1:5).eq.'-rand') THEN
            IRND=2
            write(SOUT,*) 'System random number generator'
         endif
         if (S(1:5).eq.'-help'.or.S(1:1).eq.'?') THEN
            DO J=1,9
                 write(SOUT,*) HLPOPT(J)
            ENDDO
            GOEND=1            
         endif
         if (S(1:5).eq.'-dir=') THEN
            CALL BOUNDS(S,IS,IL)
            IS=IS+5
            IL=IL-5
            IF(IL.GT.0) THEN
              RESPATH=S(IS:IS+IL-1)
              write(SOUT,*) 'dir='//S(IS:IS+IL-1)
            ENDIF  
         endif
         if (S(1:6).eq.'-exci=') THEN
            CALL BOUNDS(S,IS,IL)
            IS=IS+6
            IL=IL-6
            IF(IL.GT.0) THEN
              EXCILIB=S(IS:IS+IL-1)
              write(SOUT,*) 'exci='//S(IS:IS+IL-1)
            ENDIF  
         endif
      ENDDO      
      IF (GOEND.NE.0) GOTO 30
      SILENT=DSILENT ! set default silence leveel
      CALL RAN1SEED(ISEED)      ! Initialize random number generator
      CALL LOGO                 ! print LOGO
      CALL SETRESPATH(RESPATH)  ! set default path for configuration
      CALL READINIFILE(JOBNAME) ! read restrax.ini file
c      CALL EXCI(-999) 
c      CALL EXCI(-999,DUM4,DUM6,DUM61)  ! only test call to EXCI, set title etc...
c      CALL INITEXCI(1)
      CALL UNITS(CUNIT)         ! set units for energy 
      CALL SETPATH(' ')         ! Ask for the data path
      CALL SETCFG(CFGNAME,0)    ! Read the configuration file, but without running TRAX etc...
C// Read environment variables:
C// PGPLOT default device
      CALL GETENV('PGPLOT_DEV',S1)
      IF(S1(1:1).NE.' ') THEN
         CALL SELGRFDEV(S1,1)
      ENDIF
C// Optional plotting command, used e.g. to invoke GSView for plotting restrax.ps
      CALL GETENV('RESTRAX_PLOT',S1)
      IF(S1(1:1).NE.' ') THEN
         EXTPLOT=S1
      ELSE 
         EXTPLOT=' '
      ENDIF

      mf_cur=1
      mf_max=1

c// initialize array with gaussian random numbers
      DO I=1,MAXD
         ERHIST(I)=GASDEV1(0.,3.)
      ENDDO

C// Do initial partitioning of the histogram        
      call HISTINIT 
C// load default exci library  (without initialization)   
      call SETEXCI(EXCILIB,0)  
                
C// load parameters or a data file 
      CALL ILLNameParse(DATNAME,1) ! if name is an integer, convert to nnnnnn format (ILL convention)           
      CALL OPENFILE(' ',IRES)
      IF (IRES.LE.0.AND.RESCAL_NAME.NE.' ') THEN  ! try the filename defined in restrax.ini
         CALL OPENFILE(RESCAL_NAME,IRES)
      ENDIF
      IF (IRES.GT.0) THEN      
         IF (JOBNAME(1:1).NE.' ') THEN
           CALL REINP(JOBNAME)
           CALL LINPSETIO(SINP,SOUT,SMES)
         ENDIF  
         RETURN
      ENDIF

           
20    FORMAT(/,'Look first for a valid data filename.',/,
     * 'It should be either data in the ILL format or a RESCAL file.',/,
     * 'Look in the RESTRAX folder for a template (./demo/*.res)')
      WRITE(SMES,20) 
      WRITE(SMES,*) 
30    GOEND=1               
      END

C-----------------------------------------------------------------------------   
      SUBROUTINE RESEND
C// end of RESTRAX
C// include all actions necessary to deallocate memory etc...                
C-----------------------------------------------------------------------------  
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      
      CALL REINP(' ') ! reset input to STDIN
      CALL REOUT(' ') ! reset output to STDOUT
      CALL NESSEND    !  NESSEND must be called to deallocate
      WRITE(SMES,*) ' -> End of ResTrax'
      CALL RELEASEEXCI ! release EXCI module
      STOP 
      END
      

