C//////////////////////////////////////////////////////////////////////
C////                                                              //// 
C////  NEutron Scattering Simulation - v.2.0, (c) J.Saroun, 2000   ////
C////                                                              //// 
C//////////////////////////////////////////////////////////////////////
C////
C////  Subroutines describing objects - ALL COLLIMATOR TYPES
C////  Envelope for collimator segments of any kind:
C////  TUBE, SOLLER , GUIDE/BENDER, PARABOLIC GUIDE
C////                          
C//////////////////////////////////////////////////////////////////////

C------------------------------------
      SUBROUTINE BENDER_INIT(OBJ)
C------------------------------------	
      IMPLICIT NONE
      
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      
      IF (OBJ.TYP.EQ.2.OR.OBJ.TYP.EQ.3) THEN
         CALL PGUIDE_INIT(OBJ)
      ELSE IF (OBJ.TYP.EQ.4) THEN
         CALL EGUIDE_INIT(OBJ)
      ELSE
         CALL GUIDE_INIT(OBJ)
      ENDIF

      END        

C---------------------------------------------------------------------
      REAL*8 FUNCTION BENDER_LAM(OBJ,ID,IL,Z,IOR)
C function describing lamella profile
C ID   ... derivative
C IL   ... lamella index 
C Z    ... distance from guide entry
C IOR  ... horizontal (1) or vertical (2) 
C----------------------------------------------------------------------     
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      INTEGER*4 ID,IL,IOR
      REAL*8 PGUIDE_LAM,EGUIDE_LAM,GUIDE_LAM,Z
      
      IF (OBJ.TYP.EQ.2.OR.OBJ.TYP.EQ.3) THEN
         BENDER_LAM=PGUIDE_LAM(OBJ,ID,IL,Z,IOR)
      ELSE IF (OBJ.TYP.EQ.4) THEN
         BENDER_LAM=EGUIDE_LAM(OBJ,ID,IL,Z,IOR)
      ELSE
         BENDER_LAM=GUIDE_LAM(OBJ,ID,IL,Z,IOR)
      ENDIF
      END


C-------------------------------------------------------------------
      SUBROUTINE QUADREQ(A,B,C,X)
C Solve quadratic equation A*X^2 + B*X + C = 0
C Try to find a solution > EPS=1E-10
C 1) no solution .. return 10^30
C 2) 1 solution  .. return this
C 3) 2 solutions .. return the smaller one
C-------------------------------------------------------------------
      IMPLICIT NONE
      REAL*8 EPS
      PARAMETER (EPS=1.D-8)
      REAL*8 A,B,C,X,X1,X2,DET

      IF (A.EQ.0) THEN          
        IF (ABS(B).LT.EPS) THEN 
          GOTO 20
        ELSE
          X=-C/B
          GOTO 30
        ENDIF
      ELSE          
        DET=B**2-4*A*C
        IF (DET.EQ.0.) THEN
           X=-B/2./A
           GOTO 30
        ELSE IF (DET.LT.0.) THEN
           GOTO 20
        ELSE
           DET=SQRT(DET) 
           X1=(-B+DET)/2./A
           X2=(-B-DET)/2./A
           IF (X1.GT.EPS.AND.X2.GT.EPS) THEN
              X=MIN(X1,X2)
           ELSE IF (X1.GT.EPS) THEN
              X=X1
           ELSE IF (X2.GT.EPS) THEN
              X=X2
           ELSE 
              GOTO 20
           ENDIF
        ENDIF
      ENDIF 
                  
30    IF (X.LT.EPS) GOTO 20
      RETURN  

20    X=1.D30

      END
	
C--------------------------------------------------------
      LOGICAL*4 FUNCTION BENDER_PASS(OBJ,R,IH,IV)
c Checks, whether neutron fits inside any slit and 
c returns slit indices
C--------------------------------------------------------	
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      INTEGER*4 IH,IV
      LOGICAL*4 LOG1
      REAL*8 BENDER_LAM
      REAL*8 R(3),jh,jv,W,H
      RECORD /BENDER/ OBJ
      REAL*8 OSCD,OSCA
      COMMON /OSCBEND/ OSCD,OSCA

1     FORMAT(a10,1x,I4,1x,I4,1x,6(G11.4))

      W=2.D0*ABS(BENDER_LAM(OBJ,0,0,R(3),1))
      H=2.D0*ABS(BENDER_LAM(OBJ,0,0,R(3),2))


      jh=((R(1)-OSCD-R(3)*OSCA)/W+0.5)*OBJ.NLH
      jv=(R(2)/H+0.5)*OBJ.NLV
      IH=NINT(jh-0.5)
      IV=NINT(jv-0.5)
c      write(*,1) 'pass',IH,IV,R,jh,jv,-OSCD-R(3)*OSCA
      
      LOG1=((ABS(jh-NINT(jh))*W/OBJ.NLH.GE.OBJ.DLH/2.).AND.
     1      (ABS(jv-NINT(jv))*H/OBJ.NLV.GE.OBJ.DLV/2.).AND.
     1      (jh.GT.0.).AND.
     1      (jv.GT.0.).AND.
     1      (jh.LT.OBJ.NLH).AND.
     1      (jv.LT.OBJ.NLV))
              
      BENDER_PASS=LOG1
      END  


C---------------------------------------------------
      REAL*8 FUNCTION BENDER_REF(ID,OBJ,Q,S)
c returns reflectivity for given momentum transfer Q
C ID identifies which surface is touched
C ID=0  left 
C ID=1  right 
C ID=2  top 
C ID=3  bottom 
C---------------------------------------------------	
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      
      INTEGER*4 ID
      RECORD /BENDER/ OBJ
      REAL*8 Q,S
      INTEGER*4 iz,NR
      REAL*8 z,dQ,Q1,gamma,R	
      INTEGER*4 m_n(5)
      CHARACTER*3 m_name(5)
      REAL*8 m_alpha(128,5), m_ref1(128,5),m_ref2(128,5)            
      COMMON /MIRROR/ m_alpha,m_ref1,m_ref2,m_n,m_name    

C get critical angle, reflectivity and index to lookup table
      IF (ID.EQ.0) THEN 
        IF (S.GE.0) THEN
          gamma=OBJ.GHLU
          R=OBJ.RHLU
          NR=OBJ.NHLU
        ELSE  
          gamma=OBJ.GHLD
          R=OBJ.RHLD
          NR=OBJ.NHLD
        endif 
      ELSE IF (ID.EQ.1) THEN 
        IF (S.GE.0) THEN
          gamma=OBJ.GHRU
          R=OBJ.RHRU
          NR=OBJ.NHRU
        ELSE  
          gamma=OBJ.GHRD
          R=OBJ.RHRD
          NR=OBJ.NHRD
        ENDIF 
      ELSE IF (ID.EQ.2) THEN 
          gamma=OBJ.GVT
          R=OBJ.RVT
          NR=OBJ.NVT
      ELSE IF (ID.EQ.3) THEN 
          gamma=OBJ.GVB
          R=OBJ.RVB
          NR=OBJ.NVB
      ELSE
          gamma=0.   
          R=0. 
      ENDIF    
        
C no reflection for Q<0 or gamma=0
      IF (gamma.LE.0.OR.Q.LT.0) THEN
          BENDER_REF=0
          RETURN
      ENDIF   
        
C no lookup table, just step function for 0 < Q < 2*pi*gamma
      IF (NR.LE.0.OR.NR.GT.5) THEN
          IF (Q.GE.0.AND.Q.LT.2*PI*gamma) THEN	                         
             BENDER_REF=R
          ELSE
             BENDER_REF=0                 
          ENDIF 
      ELSE
          Q1=Q/2/PI/GammaNi
          dQ=(m_alpha(m_n(NR),NR)-m_alpha(1,NR))/(m_n(NR)-1)
          z=(Q1-m_alpha(1,NR))/dQ              
          iz=INT(z)+1
          IF (z.LT.0.OR.z.GE.m_n(NR).OR.iz.GE.m_n(NR)) THEN 
             BENDER_REF=0                 
          ELSE IF (S.GE.0) THEN
             BENDER_REF=m_ref1(iz,NR)+(z-iz+1)* 
     &            (m_ref1(iz+1,NR)-m_ref1(iz,NR))  
          ELSE IF (S.LT.0) THEN
             BENDER_REF=m_ref2(iz,NR)+(z-iz+1)*
     &            (m_ref2(iz+1,NR)-m_ref2(iz,NR))
          ELSE     
             BENDER_REF=0 
          ENDIF 
      ENDIF                         
      END
	
C----------------------------------------------------
      LOGICAL*4 FUNCTION BENDER_GO(OBJ,NEUI,NEUF)
C----------------------------------------------------	
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      
      RECORD /BENDER/ OBJ
      RECORD /NEUTRON/ NEUI,NEUF
      INTEGER*4 I
      REAL*8 V(3),K(3),R(3),P,T
      REAL*4 RAN1
      REAL*8 DJ,OSCD,OSCA
      COMMON /OSCBEND/ OSCD,OSCA
1     format(a10,2x,I8,2x,10(1x,g12.6))
      
c      dbgref=(OBJ.FRAME.NAME(1:5).EQ.'col2 ')
C Convert to local coordinate system and move to the entry
      CALL SLIT_PRE(OBJ.FRAME,NEUI.R,NEUI.K,V,K)
      
c      if (dbgref) write(*,1) OBJ.FRAME.NAME,OBJ.FRAME.COUNT,V,K

      DO I=1,2
         R(I)=V(I)-V(3)/K(3)*K(I)
      ENDDO
      R(3)=0.
      P=1.D0
      T=-V(3)/HOVM/K(3)
      
      IF (OBJ.TYP.LT.0.OR.OBJ.FRAME.SIZE(3).LE.0) GOTO 700  ! collimator ignored
      
C Make a random shift to simulate oscillating collimator
      OSCD=0
      OSCA=0
      IF (OBJ.OSCILATE.GT.0) THEN         
         dj=(RAN1()-0.5)/OBJ.NLH 
	 OSCA=dj*(OBJ.W2-OBJ.FRAME.SIZE(1))/OBJ.FRAME.SIZE(3)
	 OSCD=dj*OBJ.FRAME.SIZE(1)
      ENDIF 
      
      IF (OBJ.TYP.EQ.0) THEN
         CALL SOLLER_GO(OBJ,R,K,P,T,NEUI.S)
      ELSE IF (OBJ.TYP.EQ.1) THEN
         CALL GUIDE_GO(OBJ,R,K,P,T,NEUI.S)
      ELSE IF (OBJ.TYP.EQ.2.OR.OBJ.TYP.EQ.3) THEN
         CALL PGUIDE_GO(OBJ,R,K,P,T,NEUI.S)
      ELSE IF (OBJ.TYP.EQ.4) THEN
         CALL EGUIDE_GO(OBJ,R,K,P,T,NEUI.S)
      ELSE 
         GOTO 300
      ENDIF      
      IF (P.LE.0.D0) GOTO 300
      
700   DO I=1,3
         NEUF.R(I)=R(I)
         NEUF.K(I)=K(I)
      END DO    	   
      NEUF.P=NEUI.P*P
      NEUF.T=NEUI.T+T
      NEUF.S=NEUI.S
      OBJ.FRAME.COUNT=OBJ.FRAME.COUNT+1      
      BENDER_GO=.TRUE.
      
c      if (dbgref) write(*,1) OBJ.FRAME.NAME,OBJ.FRAME.COUNT,
c     &  NEUF.R,NEUF.K,NEUF.P
c      if (dbgref) pause
      RETURN

300   NEUF.P=0 
c      if (dbgref) write(*,1) OBJ.FRAME.NAME,OBJ.FRAME.COUNT,NEUF.P
c      if (dbgref) pause
      BENDER_GO=.FALSE.
      END        

C--------------------------------------------------------------
      SUBROUTINE SOLLER_GO(OBJ,R,K,P,T)
C GO procedure for a simple collimator (non reflecting, TYP=0)      
C INPUT:   assume R,K at the entry in local coordinates !
C RETURN:  R,K at the exit in local coordinates, P=P*transmission, T=T+passage time
C--------------------------------------------------------------	
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      
      RECORD /BENDER/ OBJ
      REAL*8 R(3),K(3),P,T
      LOGICAL*4 LOG1, BENDER_PASS
      INTEGER*4 IH,IV,IH1,IV1,I
c1     FORMAT(a10,1x,I4,1x,I4,1x,6(G11.4))

C  check passage through the entry
      LOG1=BENDER_PASS(OBJ,R,IH,IV)
c      write(*,1) OBJ.FRAME.NAME,IH,IV,R,K
      
      IF (.NOT.LOG1) GOTO 100                  
C  move to the exit
      DO I=1,2
	 R(I)=R(I)+OBJ.FRAME.SIZE(3)/K(3)*K(I)
      ENDDO        
      R(3)=OBJ.FRAME.SIZE(3)
      T=T+R(3)/HOVM/K(3)
C  check passage through the same slit at the exit
      LOG1=(LOG1.AND.BENDER_PASS(OBJ,R,IH1,IV1))       
c      write(*,1) OBJ.FRAME.NAME,IH1,IV1,R,K
      IF ((.NOT.LOG1).OR.(IH1.NE.IH).OR.(IV1.NE.IV)) GOTO 100
      RETURN
      
C no passage
100   P=0.D0
c      pause
      END        


