C//////////////////////////////////////////////////////////////////////
C////                                                              //// 
C////  NEutron Scattering Simulation - v.2.0, (c) J.Saroun, 2005   ////
C////                                                              //// 
C//////////////////////////////////////////////////////////////////////
C////
C////  Subroutines describing objects - ELLIPTIC GUIDE
C////  
C////                          
C//////////////////////////////////////////////////////////////////////

C---------------------------------------------------------------------
      SUBROUTINE EGUIDE_INIT(OBJ)
C----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      
      REAL*4 EGUIDE_A
      REAL*8 f,ZL,DIF,H1,H2,V1,V2,W
      RECORD /BENDER/ OBJ
      INTEGER*4 I
      
      CALL SLIT_INIT(OBJ.FRAME)

C limit for number of slits = 127
      IF (OBJ.NLH.GT.127) OBJ.NLH=127
      IF (OBJ.NLV.GT.127) OBJ.NLV=127
      
C HORIZONTAL      
c elliptic profile is determined by the dimensions
      H1=OBJ.FRAME.SIZE(1)
      H2=OBJ.W2
      IF (H2.EQ.H1.OR.OBJ.CH.EQ.0) THEN  !  no ellipsa, flat walls
         OBJ.CH=0.D0
      ELSE IF (H2.GT.H1) THEN
	 DIF=SQRT(H2**2-H1**2)
	 OBJ.CH=-OBJ.FRAME.SIZE(3)*H2/DIF
      ELSE 
	 DIF=SQRT(H1**2-H2**2)
	 OBJ.CH=OBJ.FRAME.SIZE(3)*H1/DIF
      ENDIF  

      f=OBJ.CH
      IF (f.EQ.0) THEN ! flat lamellas, AH & LH are angles & positions, respectively
        DO I=0,OBJ.NLH
           ZL=I*1.D0/OBJ.NLH - 0.5D0
	   OBJ.AH(I)=ZL*(H2-H1)/OBJ.FRAME.SIZE(3)
	   OBJ.LH(I)=ZL*H1
        ENDDO
      ELSE  ! elliptic lamellas, AH & LH are parameters & lengths, respectively
	w=max(H1,H2)
        DO I=0,OBJ.NLH
	     OBJ.AH(I)=EGUIDE_A(I,OBJ.NLH,w)
	     OBJ.LH(I)=OBJ.FRAME.SIZE(3)
        ENDDO
      ENDIF
      
C VERTICAL      
c elliptic profile is determined by the dimensions
      V1=OBJ.FRAME.SIZE(2)
      V2=OBJ.H2
      IF (V2.EQ.V1.OR.OBJ.CV.EQ.0) THEN  !  no parabola, flat walls
	 OBJ.CV=0.D0
      ELSE IF (V2.GT.V1) THEN
	 DIF=SQRT(V2**2-V1**2)
	 OBJ.CV=-OBJ.FRAME.SIZE(3)*V2/DIF
      ELSE
	 DIF=SQRT(V1**2-V2**2)
	 OBJ.CV=OBJ.FRAME.SIZE(3)*V1/DIF
      ENDIF  	
      f=OBJ.CV
      IF (f.EQ.0) THEN ! flat lamellas, AH & LH are angles & positions, respectively
        DO I=0,OBJ.NLV
           ZL=I*1.E0/OBJ.NLV - 0.5D0
	   OBJ.AV(I)=ZL*(V2-V1)/OBJ.FRAME.SIZE(3)
	   OBJ.LV(I)=ZL*V1
        ENDDO
      ELSE  ! elliptic lamellas, AH & LH are parameters & lengths, respectively
	w=max(V1,V2)
        DO I=0,OBJ.NLV
	     OBJ.AV(I)=EGUIDE_A(I,OBJ.NLV,w)
	     OBJ.LV(I)=OBJ.FRAME.SIZE(3)
        ENDDO
      ENDIF
      
      END

C---------------------------------------------------------------------
      REAL*8 FUNCTION EGUIDE_LAM(OBJ,ID,IL,Z,IOR)
C function describing lamella profile
C ID   ... derivative
C IL   ... lamella index 
C Z    ... distance from guide entry
C IOR  ... horizontal (1) or vertical (2) 
C----------------------------------------------------------------------     
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      
      REAL*8 a,b,zz,aa,x0
      REAL*8 Z
      INTEGER*4 ID,IL,IOR          
C//  a is the ellipse main axis // z (or slope for a planar lamellla)
C//  b is the ellipse smaller axis 
      
      IF (IOR.EQ.2) THEN   ! vertical slit
        a=OBJ.CV
        b=OBJ.AV(IL)
        x0=OBJ.LV(IL)
      ELSE                 ! horizontal slit
        a=OBJ.CH
        b=OBJ.AH(IL)
        x0=OBJ.LH(IL)
      ENDIF
      
      IF (a.LT.0) THEN     ! focal point before the guide entry
        zz=z-OBJ.FRAME.SIZE(3)	      
      ELSE    ! focal point behind the guide
        zz=z
      ENDIF
      aa=a**2-zz**2
      
      IF(a.NE.0) THEN
         IF(aa.LE.0) THEN
           EGUIDE_LAM=0.D0
	   RETURN
         ELSE
           aa=SQRT(aa)
         ENDIF
      ENDIF       

c zero deriv.      
      IF (ID.LE.0) THEN
        IF (a.EQ.0) THEN
	  EGUIDE_LAM=x0+b*z
	ELSE
          EGUIDE_LAM=b/abs(a)*aa
	ENDIF  
c 1st deriv.      
      ELSE IF (ID.EQ.1) THEN
        IF (a.EQ.0) THEN
	  EGUIDE_LAM=b
	ELSE
          EGUIDE_LAM=-b/abs(a)/aa*zz
	ENDIF  
c 2nd deriv.      
      ELSE IF (ID.EQ.2) THEN
        IF (a.EQ.0) THEN
	  EGUIDE_LAM=0.D0
	ELSE
          EGUIDE_LAM=-b*abs(a)/aa/aa**2
	ENDIF  
      ELSE
        EGUIDE_LAM=0.D0
      ENDIF
      
      END

      
C-------------------------------------------------------------------------------
      REAL*4 FUNCTION EGUIDE_A(il,nl,w)
C ellipsa parameter for il-th lamella = smaller axis 
C il ... lamella index, counted from right, right side il=0
C w  ... small axis of the outer profile 
C nl ... number of slits (number of lamellae + 1)
C sign(A) determines which side from the guide center: right/bottom(<0) or left/top(>0) 
C--------------------------------------------------------------------------------
      IMPLICIT NONE
      REAL*8 w
      INTEGER*4 il,nl
            
      EGUIDE_A=w*(il*1.E0/nl-0.5E0)
      
      END
      

C----------------------------------------------------------------------------
      REAL*8 FUNCTION EGUIDE_CROSS(kx,kz,x,z,b,a,lmax,glen)
C return path length along z-coordinate to a cross-point with elliptic surface 
C if there are 2 solutions, take the minimum one which is > 0
C return 10^30 if there is no solution 0 < result < lmax
C b      .. smaller ellipsa axis
C a      .. main ellipsa axis 
C lmax   .. limit distance (lamella length)
C x,z    .. starting point (z // guide axis)
C kx,kz  .. ray direction
C glen   .. guide length
C kz is assumed positive and >> kx
C----------------------------------------------------------------------------
      IMPLICIT NONE
      REAL*8 EPS
      PARAMETER (EPS=1.D-7)
      REAL*8 kx,kz,x,z,b,a,lmax,zz,glen   
      REAL*8 LZ,Y1,Y2,dtm,dnom,t0
            
      IF (a.LT.0) THEN     ! focal point before the guide entry
        zz=z-glen
      ELSE    ! focal point behind the guide
        zz=z
      ENDIF
      
      dnom=(kx*a)**2 + (kz*b)**2
      dtm=(a*b)**2*(dnom-(kx*zz-kz*x)**2)
      t0=x*kx*a**2 + zz*kz*b**2
      IF (dtm.LT.0.D0) THEN ! no solution
        LZ=1.D30
      ELSE   
        Y1=(-t0+SQRT(dtm))/dnom
        Y2=(-t0-SQRT(dtm))/dnom
        IF(Y1.LE.EPS.AND.Y2.LE.EPS) THEN ! no positive solution
	   LZ=1.D30
        ELSE 
          IF (Y1.GT.EPS.AND.Y2.GT.EPS) THEN ! both positive, take the smaller one
            LZ=MIN(Y1,Y2)
          ELSE ! take the positive one
            LZ=MAX(Y1,Y2)
          ENDIF             
        ENDIF   
      ENDIF  
      IF (LZ.GT.lmax) LZ=1.D30 ! no reflection behind the guide
      EGUIDE_CROSS=LZ*kz
            
      END


C---------------------------------------------------------------------
      SUBROUTINE EGUIDE_CON(OBJ,R,K,IH,IV,IC,DT,Q)
C find next contact with a slit side
C moves to the next contact point, or to the exit
C turns K, i.e. returns new K=K+Q !
C *** INPUT:
C IH,IV   ... slit indices
C R(3)    ... neutron coordinate
C K(3)    ... neutron k-vector
C *** RETURN:
C Q     .. reflection vector (magnitude)
C DT    .. time to reach the next contact point
C IC    .. which wall,  left(1), right(2), top(3), bottom (4) 
C IC=0  .. no contact, pass through
C----------------------------------------------------------------------     
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      REAL*8 R(3),K(3),Q,DT
      INTEGER*4 IH,IV,IC
      REAL*8 EGUIDE_CROSS,GUIDE_CROSS,EGUIDE_LAM
      REAL*8 kx,kz,x,z,a,f,lmax,glen
      REAL*8 LZ(4),ANG,T0,N(3)
      INTEGER*4 I
      REAL*8 OSCD,OSCA
      COMMON /OSCBEND/ OSCD,OSCA
 

c10    FORMAT(a11,1x,6(1x,G11.5),2(1x,I4),2(1x,G10.4))
c      write(*,10) 'CON START: ',R,K,IH,IV,OBJ.CH
      z=R(3)
      kz=K(3)
      
      T0=OBJ.FRAME.SIZE(3)-z ! time to the guide end
      glen=OBJ.FRAME.SIZE(3)
C HORIZONTAL RIGHT:
      f=OBJ.CH
      kx=K(1)
      lmax=T0
      IF (f.EQ.0) THEN
         a=OBJ.AH(IH)-OSCA
         x=R(1)-OBJ.DLH/2.D0-OBJ.LH(IH)-OSCD-R(3)*OSCA
         LZ(2)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=OBJ.AH(IH)
         x=R(1)-OBJ.DLH/2.D0
	 LZ(2)=EGUIDE_CROSS(kx,kz,x,z,a,f,lmax,glen)
      ENDIF
C HORIZONTAL LEFT:
      IF (f.EQ.0) THEN
         a=OBJ.AH(IH+1)-OSCA
         x=R(1)+OBJ.DLH/2.D0-OBJ.LH(IH+1)-OSCD-R(3)*OSCA
         LZ(1)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=OBJ.AH(IH+1)
         x=R(1)+OBJ.DLH/2.D0
         LZ(1)=EGUIDE_CROSS(kx,kz,x,z,a,f,lmax,glen)
      ENDIF
C VERTICAL BOTTOM:
      f=OBJ.CV
      kx=K(2)
      lmax=T0
      IF (f.EQ.0) THEN
         a=OBJ.AV(IV)
         x=R(2)-OBJ.DLV/2.D0-OBJ.LV(IV)
         LZ(4)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=OBJ.AV(IV)
         x=R(2)-OBJ.DLV/2.D0
         LZ(4)=EGUIDE_CROSS(kx,kz,x,z,a,f,lmax,glen)
      ENDIF
C VERTICAL TOP:
      IF (f.EQ.0) THEN
         a=OBJ.AV(IV+1)
         x=R(2)+OBJ.DLV/2.D0-OBJ.LV(IV+1)
         LZ(3)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=OBJ.AV(IV+1)
         x=R(2)+OBJ.DLV/2.D0
         LZ(3)=EGUIDE_CROSS(kx,kz,x,z,a,f,lmax,glen)
      ENDIF
      
c      write(*,10) 'kz*times: ',LZ(1),LZ(2)
      
      DT=MIN(LZ(1),LZ(2),LZ(3),LZ(4),T0)
      
      IC=0
      DO I=1,4
        IF (DT.EQ.LZ(I)) IC=I
      ENDDO 
      IF (IC.EQ.0) THEN ! no contact, passed through
        DT=T0
	Q=0.D0
	GOTO 50 
      ENDIF

C get the surface normal vector
      DO I=1,3
        N(I)=0.D0
      ENDDO	 
      IF (IC.EQ.2) THEN
        ANG=EGUIDE_LAM(OBJ,1,IH,Z+DT,1)
	N(3)=-ANG/SQRT((1.D0+ANG**2))
	N(1)=SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.1) THEN
        ANG=EGUIDE_LAM(OBJ,1,IH+1,Z+DT,1)
	N(3)=ANG/SQRT((1.D0+ANG**2))
	N(1)=-SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.4) THEN
        ANG=EGUIDE_LAM(OBJ,1,IV,Z+DT,2)
	N(3)=-ANG/SQRT((1.D0+ANG**2))
	N(2)=SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.3) THEN
        ANG=EGUIDE_LAM(OBJ,1,IV+1,Z+DT,2)    
	N(3)=ANG/SQRT((1.D0+ANG**2))
	N(2)=-SQRT(1.D0-N(3)**2)
      ENDIF 

c      write(*,10) 'angles: ',ANG,N
      
C scattering vector:
      Q=0.D0
      DO I=1,3
        Q=Q-K(I)*N(I)
      ENDDO	 
C move to the point of reflection:
50    DO I=1,3
        R(I)=R(I)+K(I)/K(3)*DT
      ENDDO   
C turn K-vector:
      DO I=1,3
        K(I)=K(I)+2.D0*Q*N(I)
      ENDDO   
C convert T to time units	
      DT=DT/kz/HOVM
      
c      write(*,10) 'lamellae: ',
c     & EGUIDE_LAM(OBJ,0,IH,R(3),1)+OBJ.DLH/2.D0,
c     & EGUIDE_LAM(OBJ,0,IH+1,R(3),1)-OBJ.DLH/2.D0
c      write(*,10) ' CON HIT: ',R,K,IC,OBJ.FRAME.COUNT,DT,Q
c      pause
      
      END
      
C----------------------------------------------------
      SUBROUTINE EGUIDE_GO(OBJ,R,K,P,T,S)
C GO procedure for elliptic guide (TYP=4)      
C INPUT:   assume R,K at the entry in local coordinates, i.e. R(3)=0 !
C RETURN:  R,K at the exit in local coordinates, P=P*transmission, T=T+passage time
C----------------------------------------------------	
      IMPLICIT NONE

      INCLUDE 'structures.inc'
      
      REAL*8 EPS
      PARAMETER (EPS=1.D-7)
      RECORD /BENDER/ OBJ
      REAL*8 R(3),K(3),P,T,S
      LOGICAL*4 BENDER_PASS
      REAL*8 BENDER_REF      
      INTEGER*4 IH,IV,IC,I
      REAL*8 DUM,KK,PP,TT,DT,Q
10    FORMAT(a11,1x,6(1x,G11.5),2(1x,I4),2(1x,G10.4))
      
C  check passage through the entry
      IF (.NOT.BENDER_PASS(OBJ,R,IH,IV)) GOTO 100                  

      KK=SQRT(K(1)**2+K(2)**2+K(3)**2)
      PP=1.D0
      TT=0.D0
      I=0
C  iterate through reflections
1     CALL EGUIDE_CON(OBJ,R,K,IH,IV,IC,DT,Q)
      TT=TT+DT
c debug: prevent infinite loops      
      I=I+1
      IF (I.GT.100000) THEN
         write(*,*) OBJ.FRAME.NAME,OBJ.FRAME.COUNT
	 write(*,*) 'too many iterations through bender, i=',I
         stop
      ENDIF      
      IF (IC.GT.0) THEN
         PP=PP*BENDER_REF(IC-1,OBJ,Q,S)
	 IF (PP.LT.1.D-4) GOTO 100
	 GOTO 1
      ENDIF
     
C renormalize K due to numerical errors
      DUM=SQRT(K(1)**2+K(2)**2+K(3)**2)
      DO I=1,3
        K(I)=K(I)*KK/DUM
      ENDDO
      P=P*PP
      T=T+TT
c      write(*,10) 'EXIT: ',R,K,I,OBJ.FRAME.COUNT,TT
      RETURN

100   CONTINUE
      P=0.D0
c      write(*,10) 'STOP: '
 	
      END              
      
      
      
      
