C//////////////////////////////////////////////////////////////////////
C////                                                              //// 
C////  NEutron Scattering Simulation - v.2.0, (c) J.Saroun, 2004   ////
C////                                                              //// 
C//////////////////////////////////////////////////////////////////////
C////
C////  Subroutines describing objects - GUIDE or BENDER
C////  
C////                          
C//////////////////////////////////////////////////////////////////////

C---------------------------------------------------------------------
      SUBROUTINE GUIDE_INIT(OBJ)
C----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      
      REAL*8 ZL
      RECORD /BENDER/ OBJ
      INTEGER*4 I

1     FORMAT(a,$)      
c      write(*,1) 'GUIDE_INIT '//OBJ.FRAME.NAME(1:10)
      CALL SLIT_INIT(OBJ.FRAME)

C limit for number of slits = 127
      IF (OBJ.NLH.GT.127) OBJ.NLH=127
      IF (OBJ.NLV.GT.127) OBJ.NLV=127
C set A(I) = angles for lamellae
C set L(I) = positions of lamellae at the entry 
      IF(OBJ.FRAME.SIZE(3).GT.0) THEN
      DO I=0,OBJ.NLH
         ZL=I*1.D0/OBJ.NLH - 0.5D0
	 OBJ.AH(I)=ZL*(OBJ.W2-OBJ.FRAME.SIZE(1))/OBJ.FRAME.SIZE(3)
	 OBJ.LH(I)=ZL*OBJ.FRAME.SIZE(1)
      ENDDO
      DO I=0,OBJ.NLV
         ZL=I*1.D0/OBJ.NLV - 0.5D0
	 OBJ.AV(I)=(OBJ.H2-OBJ.FRAME.SIZE(2))/OBJ.FRAME.SIZE(3)*ZL
	 OBJ.LV(I)=ZL*OBJ.FRAME.SIZE(2)
      ENDDO
      ENDIF
c      write(*,*) '... done. ',OBJ.FRAME.SIZE(3)
            
      END
      
C---------------------------------------------------------------------
      REAL*8 FUNCTION GUIDE_LAM(OBJ,ID,IL,Z,IOR)
C function describing lamella profile
C ID   ... derivative
C IL   ... lamella index 
C Z    ... distance from guide entry
C IOR  ... horizontal (1) or vertical (2) 
C----------------------------------------------------------------------     
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      
      REAL*8 a,f,x0
      REAL*8 Z
      INTEGER*4 ID,IL,IOR          
      
      IF (IOR.EQ.2) THEN   ! vertical slit
        f=OBJ.CV
        a=OBJ.AV(IL)
        x0=OBJ.LV(IL)
      ELSE                 ! horizontal slit
        f=OBJ.CH
        a=OBJ.AH(IL)
        x0=OBJ.LH(IL)
      ENDIF
      
c zero deriv.      
      IF (ID.LE.0) THEN
        GUIDE_LAM=x0+a*z+0.5*f*z**2
c 1st deriv.      
      ELSE IF (ID.EQ.1) THEN
        GUIDE_LAM=a+f*z
c 2nd deriv.      
      ELSE IF (ID.EQ.2) THEN
        GUIDE_LAM=f
      ELSE
        GUIDE_LAM=0.D0
      ENDIF
      
      END

C----------------------------------------------------------------------------
      REAL*8 FUNCTION GUIDE_CROSS(kx,kz,x,z,alpha,ro)
C return path length along z-coordinate to a cross-point with parabolic surface 
C if there are 2 solutions, take the minimum one which is > 0
C return 10^30 if there is no solution >0
C alpha  .. tan(angle between lamella and guide axis)
C ro     .. curvature
C x,z    .. starting point (z // guide axis)
C kx,kz  .. ray direction
C kz is assumed positive and >> kx
C----------------------------------------------------------------------------
      IMPLICIT NONE
      REAL*8 kx,kz,x,z,alpha,ro     
      REAL*8 LZ,A,B,C
c10    FORMAT(a10,1x,6(1x,G11.4),4(1x,I4),2(1x,G10.4))
c      write(*,10) 'cross',kx,kz,x,z,alpha,ro
      A=0.5*ro*kz**2
      B=alpha*kz-kx+ro*kz*z
      C=alpha*z+0.5*ro*z**2-x
      CALL QUADREQ(A,B,C,LZ)      
      GUIDE_CROSS=LZ*kz           
      END

C---------------------------------------------------------------------
      SUBROUTINE GUIDE_CON(OBJ,R,K,IH,IV,IC,DT,Q)
C find next contact with a slit side
C moves to the next contact point, or to the exit
C turns K, i.e. returns new K=K+Q !
C *** INPUT:
C IH,IV   ... slit indices
C R(3)    ... neutron coordinate
C K(3)    ... neutron k-vector
C *** RETURN:
C Q     .. reflection vector (magnitude)
C DT    .. time to reach the next contact point
C IC    .. which wall,  right(1), left(2), bottom (3) or top(4)
C IC=0  .. no contact, pass through
C----------------------------------------------------------------------     
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      REAL*8 R(3),K(3),Q,DT
      INTEGER*4 IH,IV,IC
      REAL*8 GUIDE_CROSS,GUIDE_LAM
      REAL*8 kx,kz,x,z,a,f
      REAL*8 LZ(4),ANG,T0,N(3)
      INTEGER*4 I
      REAL*8 OSCD,OSCA
      COMMON /OSCBEND/ OSCD,OSCA

10    FORMAT(a10,1x,6(1x,G11.4),4(1x,I4),2(1x,G10.4))
     
      z=R(3)
      kz=K(3)
      
      T0=OBJ.FRAME.SIZE(3)-z
C HORIZONTAL RIGHT:
      f=OBJ.CH
      kx=K(1)
      a=OBJ.AH(IH)-OSCA
      x=R(1)-OBJ.DLH/2.D0-OBJ.LH(IH)-OSCD-R(3)*OSCA
      LZ(2)=GUIDE_CROSS(kx,kz,x,z,a,f)
C HORIZONTAL LEFT:
      a=OBJ.AH(IH+1)-OSCA
      x=R(1)+OBJ.DLH/2.D0-OBJ.LH(IH+1)-OSCD-R(3)*OSCA
      LZ(1)=GUIDE_CROSS(kx,kz,x,z,a,f)
C VERTICAL BOTTOM:
      f=OBJ.CV
      kx=K(2)
      a=OBJ.AV(IV)
      x=R(2)-OBJ.DLV/2.D0-OBJ.LV(IV)
      LZ(4)=GUIDE_CROSS(kx,kz,x,z,a,f)
C VERTICAL TOP:
      a=OBJ.AV(IV+1)
      x=R(2)+OBJ.DLV/2.D0-OBJ.LV(IV+1)
      LZ(3)=GUIDE_CROSS(kx,kz,x,z,a,f)
      
c      write(*,10) 'times: ',LZ,T0
      
      DT=MIN(LZ(1),LZ(2),LZ(3),LZ(4),T0)
      
      IC=0
      DO I=1,4
        IF (DT.EQ.LZ(I)) IC=I
      ENDDO 
      IF (IC.EQ.0) THEN ! no contact, passed through
        DT=T0
	Q=0.D0
	GOTO 50 
      ENDIF

C get the surface normal vector
      DO I=1,3
        N(I)=0.D0
      ENDDO	 
      IF (IC.EQ.2) THEN
        ANG=GUIDE_LAM(OBJ,1,IH,Z+DT,1)
	N(3)=-ANG/SQRT((1.D0+ANG**2))
	N(1)=SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.1) THEN
        ANG=GUIDE_LAM(OBJ,1,IH+1,Z+DT,1)
	N(3)=ANG/SQRT((1.D0+ANG**2))
	N(1)=-SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.4) THEN
        ANG=GUIDE_LAM(OBJ,1,IV,Z+DT,2)
	N(3)=-ANG/SQRT((1.D0+ANG**2))
	N(2)=SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.3) THEN
        ANG=GUIDE_LAM(OBJ,1,IV+1,Z+DT,2)    
	N(3)=ANG/SQRT((1.D0+ANG**2))
	N(2)=-SQRT(1.D0-N(3)**2)
      ENDIF 

c      write(*,10) 'angles: ',ANG,N
      
C scattering vector:
      Q=0.D0
      DO I=1,3
        Q=Q-K(I)*N(I)
      ENDDO
C move to the point of reflection:
50    DO I=1,3
        R(I)=R(I)+K(I)/K(3)*DT
      ENDDO   
C turn K-vector:
      DO I=1,3
        K(I)=K(I)+2.D0*Q*N(I)
      ENDDO   
C convert T to time units	
      DT=DT/kz/HOVM
      
      END
      

C----------------------------------------------------
      SUBROUTINE GUIDE_GO(OBJ,R,K,P,T,S)
C GO procedure for a guide with flat walls (TYP=1)      
C INPUT:   assume R,K at the entry in local coordinates !
C RETURN:  R,K at the exit in local coordinates, P=P*transmission, T=T+passage time
C----------------------------------------------------	
      IMPLICIT NONE

      INCLUDE 'structures.inc'
      
      RECORD /BENDER/ OBJ
      REAL*8 R(3),K(3),P,T,S
      LOGICAL*4  BENDER_PASS
      REAL*8 BENDER_REF      
      INTEGER*4 IH,IV,I,IC
      REAL*8 DUM,KK,PP,TT,DT,Q,CO,SI,BETA,DELTA,R2(3),K2(3)

c10    FORMAT(a10,1x,6(1x,G11.4),4(1x,I4),2(1x,G10.4))
c11    FORMAT(a10,1x,8(1x,G11.4))
            
C  check passage through the entry
      IF (.NOT.BENDER_PASS(OBJ,R,IH,IV)) GOTO 100                  

      KK=SQRT(K(1)**2+K(2)**2+K(3)**2)
      PP=1.D0
      TT=0.D0
      I=0
c      write(*,10) 'start',R,K

C  iterate through reflections
1     CALL GUIDE_CON(OBJ,R,K,IH,IV,IC,DT,Q)
c      write(*,10) 'bounce',R,K,IH,IV,IC,I,DT,Q
      TT=TT+DT
c debug: prevent infinite loops      
      I=I+1
      IF (I.GT.100000) THEN
         write(*,*) OBJ.FRAME.NAME,OBJ.FRAME.COUNT
	 write(*,*) 'too many iterations through bender, i=',I
         stop
      ENDIF      
      IF (IC.GT.0) THEN
         PP=PP*BENDER_REF(IC-1,OBJ,Q,S)
	 IF (PP.LT.1.D-4) GOTO 100
	 GOTO 1
      ENDIF

C  HORIZONTAL correction for beam deflection
      BETA=OBJ.CH*OBJ.FRAME.SIZE(3)
      IF (BETA.NE.0) THEN  
         DELTA=0.5*OBJ.CH*OBJ.FRAME.SIZE(3)**2
	 SI=BETA/SQRT(1.D0+BETA**2)    
	 CO=SQRT(1.D0-SI**2)
	 R2(1)=(R(1)-DELTA)*CO-(R(3)-OBJ.FRAME.SIZE(3))*SI
         R2(2)=R(2)
         R2(3)=(R(3)-OBJ.FRAME.SIZE(3))*CO+(R(1)-DELTA)*SI
	 K2(1)=K(1)*CO-K(3)*SI
         K2(2)=K(2)
         K2(3)=K(3)*CO+K(1)*SI
         DO I=1,3
           R(I)=R2(I)
           K(I)=K2(I)
         END DO 
	 R(3)=R(3)+OBJ.FRAME.SIZE(3)
      ENDIF

C  VERTICAL correction for beam deflection
      BETA=OBJ.CV*OBJ.FRAME.SIZE(3)
      IF (BETA.NE.0) THEN  
         DELTA=0.5*OBJ.CV*OBJ.FRAME.SIZE(3)**2
	 SI=BETA/SQRT(1.D0+BETA**2)    
	 CO=SQRT(1.D0-SI**2)
	 R2(2)=(R(2)-DELTA)*CO-(R(3)-OBJ.FRAME.SIZE(3))*SI
         R2(1)=R(1)
         R2(3)=(R(3)-OBJ.FRAME.SIZE(3))*CO+(R(2)-DELTA)*SI
	 K2(2)=K(2)*CO-K(3)*SI
         K2(1)=K(1)
         K2(3)=K(3)*CO+K(2)*SI
         DO I=1,3
           R(I)=R2(I)
           K(I)=K2(I)
         END DO 
	 R(3)=R(3)+OBJ.FRAME.SIZE(3)
      ENDIF
      
c      write(*,11) 'OK',R,K,PP,TT
c      pause
     
C renormalize K due to numerical errors
      DUM=SQRT(K(1)**2+K(2)**2+K(3)**2)
      DO I=1,3
        K(I)=K(I)*KK/DUM
      ENDDO
      P=P*PP
      T=T+TT
      RETURN

100   CONTINUE
c      write(*,11) 'failed',R,K,PP,TT
c      pause
      P=0.D0
 	
      END              
      

