C//////////////////////////////////////////////////////////////////////
C////                                                              //// 
C////  NEutron Scattering Simulation - v.2.0, (c) J.Saroun, 2000   ////
C////                                                              //// 
C//////////////////////////////////////////////////////////////////////
C////
C////  Subroutines describing objects - CRYSTAL ARRAY
C////  
C////                          
C//////////////////////////////////////////////////////////////////////
C ---------------------------------------
      SUBROUTINE CRYST_INIT2(CR)
C ---------------------------------------
      implicit none
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'                        
      RECORD /CRYSTAL/ CR
      REAL*8 Z,GETQKIN,GETREFDYN,GETMI
      INTEGER*4 I,J

C///  THB,DHKL,etc.. must be specified before !

      CALL SLIT_INIT(CR.FRAME)
        
      CR.GTOT=2*PI/ABS(CR.DHKL)
      CR.G(1)=CR.GTOT*SIN(CR.CHI)
      CR.G(2)=0.
      CR.G(3)=CR.GTOT*COS(CR.CHI)
      CR.STMCH=SIN(CR.THB-CR.CHI)
      CR.CTMCH=COS(CR.THB-CR.CHI)
      CR.LAMBDA=2.*CR.DHKL*SIN(CR.THB)
      DO I=1,3
        CR.MAPG(I)=.FALSE.
        DO J=1,3 
          CR.DG_DR(I,J)=0
        ENDDO
      ENDDO               

c G-gradient for elastically bent crystal      
      IF(CR.HMOS.LT.sec) THEN 
          CR.DG_DR(1,1)=-COS(CR.CHI)*CR.GTOT*CR.RH
          CR.DG_DR(1,3)=SIN(CR.CHI)*CR.GTOT*CR.RH
          CR.DG_DR(3,1)=SIN(CR.CHI)*CR.GTOT*CR.RH
          CR.DG_DR(2,2)=0    ! no vertical bending
          CR.DG_DR(3,3)=-CR.POI*COS(CR.CHI)*CR.GTOT*CR.RH
          CR.MAPG(1)=.TRUE.        
          CR.MAPG(3)=.TRUE.        
      ENDIF
c d-gradient
      IF(CR.DGR.NE.0.) THEN         
          Z=1.D-4*CR.GTOT*CR.DGR
          CR.DG_DR(1,1)=CR.DG_DR(1,1)+Z*cos(CR.DGA+CR.CHI)
          CR.DG_DR(1,3)=CR.DG_DR(1,3)-Z*sin(CR.DGA+CR.CHI)
          CR.DG_DR(3,1)=CR.DG_DR(3,1)-Z*sin(CR.DGA+CR.CHI)
          CR.DG_DR(3,3)=CR.DG_DR(3,3)-Z*cos(CR.DGA+CR.CHI)
          CR.MAPG(1)=.TRUE.        
          CR.MAPG(3)=.TRUE.        
      ENDIF                  
c unit vector |- to G
      CR.gama(1)=COS(CR.CHI)
      CR.gama(3)=-SIN(CR.CHI)
      CR.gama(2)=0 

      CR.QHKL=GETQKIN(CR,CR.lambda)  ! kin. reflectivity
      CR.dext=CR.DHKL/CR.LAMBDA*SQRT(4*PI/CR.QML) ! extinction length
      Z=CR.dlam/CR.Dext
      IF(Z.GT.1D-5.AND.CR.HMOS.GT.SEC) THEN 
         CR.Ext1=TANH(Z)/Z         ! primary extinction
      ELSE
         CR.Ext1=1.
      ENDIF         
      CR.MI=GETMI(CR,CR.lambda,300.D0)  ! absorption coefficient
      CR.REF=GETREFDYN(CR,CR.LAMBDA)    ! peak reflectivity (no mosaic)         
      CR.DELTA=CR.QHKL*CR.Dext*1D-4/PI   ! Darwin box width
      IF(CR.HMOS.GT.SEC) CR.HMOS=MAX(CR.HMOS,CR.DELTA)
      
      END


C        -------------------------------------------
      LOGICAL FUNCTION CRYST_GO2(CRYST,NEUI,NEUF)
C        -------------------------------------------
      implicit none
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INCLUDE 'randvars.inc'
                        
      RECORD /CRYSTAL/ CRYST
      RECORD /NEUTRON/ NEUI,NEUF
      REAL*8 V(3),K(3),R(3),KaG(3)
      REAL*8 PP,DT,H,TIN,TOUT
      real*8 CRYST_ARRAY
      INTEGER*4 i

c      logical*4 ref  
c      ref=(cryst.frame.size(2).eq.80.and.cryst.frame.count.gt.1000)      
1     format(a10,2x,I8,2x,10(1x,g12.6))
c2     format(a10,(1x,g12.6),5x,$)
      IF(DBGREF) then
c        write(*,1) 'XRND ',(XRND(i),i=1,5)
      write(*,1) CRYST.FRAME.NAME,CRYST.FRAME.COUNT,NEUI.R,NEUI.K,NEUI.P
      endif  

C/// if nh=0 then accept neutrons without transformations
      IF (CRYST.NH.EQ.0) then           
        neuf=neui
        dt=(cryst.frame.dist-neui.r(3))/neui.k(3)
        do i=1,3
              neuf.r(i)=neui.r(i)+dt*neui.k(i)
        end do
        NEUF.T=NEUI.T+DT/HOVM
        GOTO 10
      endif   

      if (CRYST.MAG*NEUI.S.LT.0) GOTO 99      

c      if (ref) then
c         call wrtneu(NEUI)
c         write(*,1) 'dAlpha1: ',atan(NEUI.K(1)/NEUI.K(3))/deg,NEUI.P
c      endif   
        
      CALL SLIT_PRE(CRYST.FRAME,NEUI.R,NEUI.K,V,K)
      V(1)=V(1)+CMX              
C test right barrier
      IF (CBAR.GT.0) THEN
         H=V(3)-(CRYST.FRAME.SIZE(1)/2.+V(1)+40.)/K(1)*K(3)
         IF (ABS(H).LT.CBAR.AND.K(1).GT.0) GOTO 99
      ENDIF
      
C/// move to the entry point          
      call CR_BORDER(CRYST,V,K,TIN,TOUT)            
      IF (TIN.GE.TOUT) GOTO 99     ! No intersection with the crystal
      DO I=1,3
         V(I)=V(I)+(TIN+1.D-7*(TOUT-TIN))*K(I)
      END DO

c      if (ref) write(*,1) 'start: ',(V(I),i=1,3),NEUI.P

      PP=CRYST_ARRAY(CRYST,V,K,R,KaG,DT)
      
      IF (PP.GT.1.D-4) THEN
C test right barrier
          IF (CBAR.GT.0) THEN
            H=R(3)-(CRYST.FRAME.SIZE(1)/2.+R(1)+40.)/KaG(1)*KaG(3)
            IF (ABS(H).LT.CBAR.AND.KaG(1).LT.0) GOTO 99
          ENDIF
C transform to local axis coordinate and return
          R(1)=R(1)-CMX                                 
          NEUF.S=NEUI.S
          NEUF.T=NEUF.T+DT/HOVM
          NEUF.P=NEUI.P*PP
c      if (ref) write(*,1) 'end: ',(R(I),i=1,3),NEUF.P
          CALL SLIT_POST(CRYST.FRAME,R,KaG,NEUF.R,NEUF.K)
c      if (ref) then
c         call wrtneu(NEUF)
c         write(*,1) 'dAlpha2: ',
c     *              (atan(NEUF.K(1)/NEUF.K(3))+2*CRYST.THB)/deg,NEUF.P
c      endif   
          CRYST.FRAME.COUNT=CRYST.FRAME.COUNT+1
          GOTO 10
      ELSE
          GOTO 99
      ENDIF
        
10    CRYST_GO2=.TRUE.
c      if (dbgref)  then
c      write(*,1) CRYST.FRAME.NAME,CRYST.FRAME.COUNT,NEUF.R,NEUF.K,NEUF.P
c      endif
       RETURN
      
99    NEUF.P=0
c      if (dbgref)  then
c      write(*,1) CRYST.FRAME.NAME,CRYST.FRAME.COUNT,NEUF.P
c      endif
      CRYST_GO2=.FALSE.
      RETURN        
      END        
        

C---------------------------------------------------------------
      REAL*8 FUNCTION HMOS_DIST(X)
C Distribution of horizontal angular deviation of mosaic blocks
C --------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      REAL*8 X,SQRT2PI,SQRT2LN2,DD0
      PARAMETER (SQRT2PI=2.506628274631, SQRT2LN2=1.177410023, 
     *           DD0=0.270347526)      

      IF (MDIST.EQ.1) THEN        
C pseudo-Voigt
        HMOS_DIST=0.5*DD0/(1. + (X/SQRT2LN2)**2)+
     *   0.5*EXP(-0.5*X**2)/SQRT2PI
      ELSE IF (MDIST.EQ.2) THEN        
C Lorenz
        HMOS_DIST=DD0/(1. + (X/SQRT2LN2)**2)
      ELSE IF (MDIST.EQ.3) THEN        
C Rectangle
        IF (ABS(X).LE.0.5) THEN
          HMOS_DIST=1.
        ELSE
          HMOS_DIST=0. 
        ENDIF   
      ELSE 
C Gauss
          HMOS_DIST=EXP(-0.5*X**2)/SQRT2PI
      ENDIF
      END 

C-----------------------------------------------------------
      REAL*8 FUNCTION CRYST_ARRAY(CR,R1,K1,R2,K2,DT)
C Transmission function for mosaic crystal 
C (eta>")  
C simplified version (Darwin width=0)    
C-----------------------------------------------------------
      implicit none
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'crystal.inc'
      REAL*8 EPS
      PARAMETER (EPS=1D-5)
      RECORD /CRYSTAL/ CR
      REAL*8 R1(3),K1(3),R2(3),K2(3),DT
      REAL*8 G(3)
      INTEGER*4 I
      REAL*8 PP,TIN,TOUT
      REAL*8 KK,K0,Qhkl,Z,MI

c      logical*4 ref      
c      ref=(cr.frame.size(2).eq.81.and.cr.frame.count.gt.1000)      
1     format(a,10(1x,g12.6))
      

      PP=1.D0
C/// calculate values, which are constant in the course of the iteration      
      KK=K1(1)**2+K1(2)**2+K1(3)**2
      K0=SQRT(KK)      
      Qhkl=CR.Qhkl/10.*CR.DW*CR.Ext1
      MI=CR.MI/10.
      PATH=0.D0
      TOF=0.D0
      call CR_BORDER(CR,R1,K1,TIN,TOUT)
            
      IF (TIN.GE.TOUT) GOTO 99     ! No intersection with the crystal
C/// move to the entry point = 1st segment sector entry         
      DO I=1,3
         R2(I)=R1(I)+(TIN+1.D-7*(TOUT-TIN))*K1(I)
      END DO
      
C ****  Begin of multiple scattering iteration cycle  ****

C// generate random walk step in K direction
C---------------------------------------------------
50    CALL WALKSTEP(CR,1,R2,K1,K0,Qhkl,G,PP,TOUT)     
      if (PP.LT.EPS) goto 99   ! don't follow trajectories with low probability         
C// move to the new reflection point and set K=K+G
      DO I=1,3
         K2(I)=K1(I)+G(I)
      END DO
      
      Z=SQRT(KK/(K2(1)**2+K2(2)**2+K2(3)**2))  ! set |K2| = |K1|
      DO I=1,3
           K2(I)=K2(I)*Z
      END DO
      
c      if (dbgref) then
c        write(*,1) 'dTheta1: ',atan(G(1)/G(3))/deg
c        write(*,1) 'step1:  ',(r2(i),i=1,3),PP,
c     *              (atan(-k1(3)/k1(1))-CR.THB)/deg,
c     *              (atan(k2(3)/k2(1))-CR.THB)/deg
c        pause
c      endif  

C// generate 2nd random walk step in K+G direction
C-------------------------------------------------
      CALL WALKSTEP(CR,-1,R2,K2,K0,Qhkl,G,PP,TOUT)      
      
      IF (PP.LT.EPS) GOTO 99
      if (TOUT.LE.0) THEN    
        PP=PP*EXP(-MI*PATH)
      else       
        DO I=1,3
           K1(I)=K2(I)+G(I)
        END DO
        Z=SQRT(KK/(K1(1)**2+K1(2)**2+K1(3)**2))  ! set |K2| = |K1|
        DO I=1,3
           K1(I)=K1(I)*Z
        END DO        
        
c        if (dbgref) then
c          write(*,1) 'dTheta2: ',atan(G(1)/G(3))/deg
c          write(*,1) 'step2:  ',(r2(i),i=1,3),PP,
c     *              (atan(-k1(3)/k1(1))-CR.THB)/deg,
c     *              (atan(k2(3)/k2(1))-CR.THB)/deg
c          pause
c        endif
        
         GOTO 50  ! repeat cycle
      endif
        
C ****  End of multiple scattering cycle  ****
      
c      if (ref) then
c        write(*,1) 'end:  ',(r2(i),i=1,3),pp
c        write(*,1) 'path: ',path,tof*k0,pp
c        pause  
c      endif
      
      DT=TOF
      CRYST_ARRAY=PP
      RETURN
99    CRYST_ARRAY=0.D0
c      IF (DBGREF) THEN
c        WRITE(*,1) 'EX:   ',PP,TOUT*K1(3)
C        pause     
c      ENDIF
      END         


C--------------------------------------------------------------
      SUBROUTINE WALKSTEP(CR,DIR,R,K,K0,Q,G,PP,TOUT)
C Generate random walk step in the crystal
C//   Q = Qhkl*DW*Ext1
C//   K0=|K|
C//   R,K position and K-vector
C//   DIR=1,-1 for directions K,K+G
C//   TOUT  .Time to reach assembly exit
C//   G(3) ...Local diffraction vector at R
C//   PP ... weight to be attributed to the event
C--------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      INCLUDE 'crystal.inc'
      RECORD /CRYSTAL/ CR
      REAL*8 EPS
      PARAMETER(EPS=1D-5)
      INTEGER*4 I,M,DIR,I0(3)
      REAL*8 Z,PHI,P,DT,K0,Q,TOUT,PP
      REAL*8 R(3),K(3),R0(3),G(3)
      REAL*4 RAN1,GASDEV1
      
c      logical*4 ref      
c      ref=(cr.frame.size(2).eq.81.and.cr.frame.count.gt.1000)      
c1     format(a,10(1x,g12.6))


      
C add random vertical mosaic angle 
      PHI=CR.VMOS*GASDEV1(1,0.,3.)       
C trace through segments and find times, scatt. probabilities, angular dev. etc..
      call CR_BORDERS(CR,DIR,R,K,K0,Q,PHI) 
      
      IF (NSEG.LE.0) GOTO 99
            
      TOUT=TSEG(NSEG) ! time to the last segment exit
      P=PSEG(NSEG)   ! scattering probability up to the assembly exit 
      IF(DIR.LT.0) P=1. ! for K+G direction, consider the possibility of leaving the assembly  
      IF (P*PP.LT.EPS) GOTO 99 ! don't follow trajectories with low probability
C Select a segment for next reflection according to the scattering probabilities
      Z=P*RAN1()
      M=1
      DO WHILE (M.LE.NSEG.AND.Z.GE.PSEG(M))
          M=M+1
      END DO
      
C No reflection in any segment
      IF(M.GT.NSEG) THEN
        IF(DIR.LT.0) THEN
            GOTO 90 ! go to the last segment exit and return
        ELSE    
            GOTO 99 ! event stopped (no reflection) 
        ENDIF   
      ENDIF          

C Find point of reflection inside the M-th segment 
      CALL SEGTRACE(CR,K0,Q,M,DT)  
      
      IF (DT.LE.0) GOTO 99

C accumulate flight-path through the material       
      DO I=1,M-1
         PATH=PATH+(TSEG(I)-TSEG0(I))*K0
      ENDDO
      PATH=PATH+DT*K0

C accumulate time-of-flight     
      TOF=TOF+DT+TSEG0(M)
                  
C Move to the point of reflection
      DO I=1,3
         R(I)=R(I)+(DT+TSEG0(M))*K(I)
      ENDDO      
      
C      IF(REF) THEN       
C       write(*,1) 'DIR: ',DIR
C       write(*,1) 'DT : ',DT*K0,TSEG0(M)*K0       
C       write(*,1) 'R1: ',(R(I)-(DT+TSEG0(M))*K(I),I=1,3)
C       write(*,1) 'R2: ',(R(I),I=1,3)
C      pause
C      ENDIF
      

C Coordinates of the segment centre
      CALL SEGCOORD(CR,R,R0,I0)
        
c      if (ref)  write(*,1) 'segment  : ',I0(1)
C Calculate local G-vector 
      CALL LOCALG(CR,DIR,R,R0,I0,ALPHA(M)+GRAD(M)*DT,PHI,G)
      
C      IF (REF) THEN        
C        Z=v3xv3(G,G)
C        DO I=1,3
C          AT(I)=K(I)+G(I)
C        ENDDO
C        write(*,1) 'K  : ',(K(I),I=1,3)
C        write(*,1) 'G  : ',(G(I),I=1,3)
C        write(*,1) 'DELTA: ',2.*v3xv3(AT,G)-Z        
C        CALL LOCALG(CR,-DIR,R,R0,I0,ALPHA(M)+GRAD(M)*DT,PHI,G0)
C        write(*,1) 'K  : ',(AT(I),I=1,3)
C        write(*,1) 'G  : ',(G0(I),I=1,3)
C        Z=v3xv3(G0,G0)
C        DO I=1,3
C          AT(I)=AT(I)+G0(I)
C        ENDDO
C        write(*,1) 'DELTA: ',2.*v3xv3(AT,G0)-Z
C      ENDIF
      
      PP=PP*P
      RETURN
            
C No reflection when going along K+G (DIR<0) 
90    DO I=1,3
         R(I)=R(I)+TSEG(NSEG)*K(I)
      ENDDO
      TOF=TOF+TSEG(NSEG)
      DO I=1,NSEG
          PATH=PATH+(TSEG(I)-TSEG0(I))*K0
      ENDDO
      TOUT=0.D0
      
      RETURN
      
C Left without reflection 
99    PP=0.D0
      TOUT=0.D0
            
      END

C ---------------------------------------------------------------------C
      SUBROUTINE CR_BORDERS(CR,DIR,R,K,K0,Q,PHI)
C Traces through all segments along the neutron path
C PHI= vertical mosaic tilt. angle 
C Q = Qhkl*DW*Ext1
C K0=|K|
C R,K position and K-vector
C DIR=1,-1 for directions K,K+G
C IN /CRBORDERS/ returns for each segment crossed:
C TSEG    time to the I-th segment exit, started at the assembly entry
C TSEG0   time to the I-th segment entry, -"-
C ALPHA   dThetaB, deviation from Bragg angle 
C GRAD    grad(ThetaB), gradient of Bragg angle along K 
C PSEG    scattering probability
C NSEG    number of crossed segments
C PATH    accumulates path-length through the material
C ---------------------------------------------------------------------C
      implicit none
      INCLUDE 'structures.inc'
      INCLUDE 'crystal.inc'
      RECORD /CRYSTAL/ CR
      INTEGER*4 I,DIR,J,I0(3)
      REAL*8 KK,K0,T,DT,Q,PHI,TIN1,TIN2,TOUT1,TOUT2     
      REAL*8 R(3),K(3),V(3),V0(3)
      REAL*8 LIMIT
      PARAMETER(LIMIT=0.5D0)
C      LOGICAL*4 REF
C      REF=(CR.FRAME.COUNT.GE.10000)      
      
C1     FORMAT(a11,6(1x,G16.10))

      J=0
      TSEG(0)=0.D0
      TSEG0(0)=0.D0
      PSEG(0)=0.D0
      T=0.D0 ! T measures time-fo-flight
      KK=K0**2
      DO I=1,3
         V(I)=R(I)
      END DO
C      if(ref)  write(*,1) 'START AT: ',(V(I),I=1,3),DIR
      
      DO WHILE ((ABS(V(1)/CR.FRAME.SIZE(1)).LT.LIMIT).AND.
     *          (ABS(V(2)/CR.FRAME.SIZE(2)).LT.LIMIT).AND.
     *          (ABS(V(3)/CR.FRAME.SIZE(3)).LT.LIMIT).AND.J.LT.MSEG)
      
        CALL SEGCOORD(CR,V,V0,I0) ! get segment coordinates
        
C       if(ref)  write(*,1) 'CENTER: ',(V0(I),I=1,3),(I0(I),I=1,3)
         
C Get entry (TIN) and exit (TOUT) times of a neutron moving along K and starting at V
C (1) .. crossing segment sector
C (2) .. crossing the segment itself (differs by gaps between the segments, CR.DH ...)
        CALL SEGCROSS(CR,V,K,V0,TIN1,TIN2,TOUT1,TOUT2)

C       if (ref) then
C         write(*,1) 'CROSS IN : ',(TIN1*K(I),I=1,3),(TIN2*K(I),I=1,3)
C         write(*,1) 'CROSS OUT: ',(TOUT1*K(I),I=1,3),(TOUT2*K(I),I=1,3)
C       endif  
                          
C Count only intersected segments
        IF((TOUT2-TIN2)*K0.GT.1.D-3.AND.TOUT2.GT.1.D-10) THEN ! must start before or inside the segment
          J=J+1        
C Move to the segment entry, if not already inside   
          IF (TIN2.GT.0.) THEN 
            DO I=1,3
              V(I)=V(I)+TIN2*K(I)
            ENDDO 
            T=T+TIN2
          ELSE
            TIN2=0.D0
          ENDIF
          TSEG0(J)=T     ! J-th segment starting time (with respect to the orig. position R)    
          TSEG(J)=T+TOUT2-TIN2  ! J-th segment exit time     
                          
C Calculate total scattering probability for the segment, starting at V          
          CALL SEGSCATT(CR,J,DIR,V,K,PHI,Q,TSEG(J)-TSEG0(J))
          
C       if(ref) THEN
C          write(*,1) 'SEG POS : ',(V(I),I=1,3)
C          write(*,1) 'SEG EXIT: ',(V(I)+K(I)*(TSEG(J)-TSEG0(J)),I=1,3)
C          write(*,1) 'ALPHA: ',ALPHA(J),GRAD(J),-ALPHA(J)/GRAD(J)*K0
C          write(*,1) 'PSEG : ',PSEG0(J),PSEG(J)
C       endif

        ELSE
          TIN2=0.D0        
        ENDIF

C Move to the entry of the next segment sector        
        DT=1.000001D0*(TOUT1-TIN2)  ! move slightly behind to have it realy INSIDE the next sector
        DO I=1,3
           V(I)=V(I)+DT*K(I)
        ENDDO         
        T=T+DT   
           
C        if(ref) write(*,1) 'NEXT SEG: ',J,(V(I),I=1,3) 
      
      ENDDO
                  
      NSEG=J
      
      END
      

C -------------------------------------------------------------
      SUBROUTINE LOCALG(CR,DIR,R,R0,I0,ETA,PHI,G)
C Calculate local G-vector at R
C I0(3) segment coordinates
C R0(3) segment center physical coordinates
C ETA .. horizontal tilt angle of mosaic domain
C PHI .. vertical tilt angle of mosaic domain
C -------------------------------------------------------------
      implicit none
      INCLUDE 'structures.inc'
      RECORD /CRYSTAL/ CR
      INTEGER*4 I,DIR,I1,I0(3)
      REAL*8 R(3),R0(3),G(3),W(3),AT(3),G0(3)
      REAL*8 Z,ETA,PHI,GABS
      
C Calculate local G-vector (only G-gradient) 
      DO I=1,3
          G0(I)=CR.G(I) 
          IF (CR.MAPG(I)) THEN
              W(I)=0.D0
            DO I1=1,3
                W(I)=W(I)+CR.DG_DR(I,I1)*(R(I1)-R0(I1))
            ENDDO
            G0(I)=G0(I)+W(I)
          ENDIF  
      END DO
      GABS=SQRT(G0(1)**2+G0(2)**2+G0(3)**2)

C Add segment tilt angle and vertical mosaic spread 
      CALL SEGTILT(CR,I0,AT)
      G(1)=G0(1)-G0(3)*(AT(1)+AT(3))  
      G(3)=G0(3)+G0(1)*(AT(1)+AT(3)) 
      G(2)=G0(2)-G0(3)*AT(2)+GABS*PHI

C Add the angle of the mosaic block
      DO I=1,3
         G(I)=G(I)+GABS*CR.GAMA(I)*ETA
      END DO
C Renormalize
      Z=GABS/SQRT(G(1)**2+G(2)**2+G(3)**2)
      DO i=1,3
         G(i)=DIR*G(i)*Z
      ENDDO
      END


C -------------------------------------------------------------
      SUBROUTINE CR_BORDER(CR,R,K,TIN,TOUT)
C     Returns times of intersection with the crystal assembly borders, 
C     started at current position R and measured along K.
C     All in crystal local coordinate.
C  !! Time is in units [sec*h/m] everywhere   i.e. length=time*K
C--------------------------------------------------------------
      implicit none
      INCLUDE 'structures.inc'
c      INCLUDE 'crystal.inc'
      RECORD /CRYSTAL/ CR
        
      REAL*8 R(3),K(3),TIN,TOUT,T1(3),T2(3),DUM
      INTEGER*4 I
        
      DO I=1,3
      IF (ABS(K(I)).GT.1.0D-8) THEN
               T2(I)=(CR.FRAME.SIZE(I)/2.D0 - R(I))/K(I)
        T1(I)=(-CR.FRAME.SIZE(I)/2.D0 - R(I))/K(I)
        IF (T1(I).GT.T2(I)) THEN
          DUM=T1(I)
          T1(I)=T2(I)
          T2(I)=DUM
        ENDIF            
      ELSE
          T2(I)=1.0D30
          T1(I)=-1.0D30
      ENDIF
      END DO
      TIN=MAX(T1(1),T1(2),T1(3))
      TOUT=MIN(T2(1),T2(2),T2(3))
      IF (TIN.GT.TOUT) THEN
         TIN=1D30
         TOUT=1D30
      ENDIF
      END            



C ----------------------------------------------------------
      SUBROUTINE SEGTILT(CR,I0,AT)
C     return tilt angles
C ----------------------------------------------------------
      implicit none
      INCLUDE 'structures.inc'
      RECORD /CRYSTAL/ CR
      REAL*8 AT(3),da
      INTEGER*4 I0(3)
      
      IF(I0(1).GT.0.AND.I0(1).LE.CR.NH) THEN
           da=CR.FRAME.SIZE(1)*CR.RH/CR.nh        
           AT(1)=(-(CR.nh-1)/2.D0+I0(1)-1.D0)*da
      ELSE 
           AT(1)=0.D0
      ENDIF
      IF(I0(2).GT.0.AND.I0(2).LE.CR.NV) THEN
           da=CR.FRAME.SIZE(2)*CR.RV/CR.nv        
           AT(2)=(-(CR.nv-1)/2.D0+I0(2)-1.D0)*da
      ELSE 
           AT(2)=0.D0
      ENDIF
      IF(I0(3).GT.0.AND.I0(3).LE.CR.NB) THEN
           da=CR.FRAME.SIZE(3)*CR.RB/CR.nb        
           AT(3)=(-(CR.nb-1)/2.D0+I0(3)-1.D0)*da
      ELSE 
           AT(3)=0.D0
      ENDIF
      
c      IF(CR.HMOS.GT.0) THEN
c      write(*,*) CR.FRAME.NAME,CR.NH,CR.NV,CR.NB
c1     format (3(G10.4,2x))      
c      write(*,*) (I0(I),I=1,3)
c      write(*,1) (AT(I),I=1,3)
c     pause      
c      ENDIF
      END
      
      
C ---------------------------------------------------------------------------
      SUBROUTINE SEGCOORD(CR,R,R0,I0)
C     return coordinates of the segment R0, in which the particle at R resides
C ---------------------------------------------------------------------------
      implicit none
      INCLUDE 'structures.inc'
      RECORD /CRYSTAL/ CR
      REAL*8 R(3),R0(3),HALF
      INTEGER*4 I0(3) 
      PARAMETER(HALF=0.5D0)

C ih,iv,ib are the integre-coordinates of the closest segment
      I0(1)=NINT((R(1)/CR.FRAME.SIZE(1)+HALF)*CR.NH+HALF)
      I0(2)=NINT((R(2)/CR.FRAME.SIZE(2)+HALF)*CR.NV+HALF)
      I0(3)=NINT((R(3)/CR.FRAME.SIZE(3)+HALF)*CR.NB+HALF)
C get physical coordinates of the segment center
      R0(1)=CR.FRAME.SIZE(1)*(1.D0*(I0(1)-HALF)/CR.NH-HALF)
      R0(2)=CR.FRAME.SIZE(2)*(1.D0*(I0(2)-HALF)/CR.NV-HALF)
      R0(3)=CR.FRAME.SIZE(3)*(1.D0*(I0(3)-HALF)/CR.NB-HALF)
      END


C ---------------------------------------------------------------------------C
      SUBROUTINE SEGCROSS(CR,R,K,R0,TIN1,TIN2,TOUT1,TOUT2)
C     return times when the particle (R,K) crossess the boarders
C of the segment sector (TIN1,TOUT1) and the segment itself (TIN2,TOUT2) 
C ---------------------------------------------------------------------------C
      implicit none
      INCLUDE 'structures.inc'
      RECORD /CRYSTAL/ CR
      REAL*8 R(3),R0(3),K(3),TIN1,TIN2,TOUT1,TOUT2
      REAL*8 R1(3),BSZ(3),T1(3),T2(3),DUM
      INTEGER*4 I 
1     FORMAT(a10,6(2x,G12.6))

C sector size
      BSZ(1)=CR.FRAME.SIZE(1)/CR.NH
      BSZ(2)=CR.FRAME.SIZE(2)/CR.NV
      BSZ(3)=CR.FRAME.SIZE(3)/CR.NB
C coordinates relative to segment centre
      DO I=1,3      
        R1(I)=R(I)-R0(I)
      ENDDO
C Get entry and exit times for segment sector      
      DO I=1,3      
      IF (ABS(K(I)).GT.1.0D-10) THEN
               T2(I)=(BSZ(I)/2.-R1(I))/K(I)
        T1(I)=(-BSZ(I)/2.-R1(I))/K(I)
        IF (T1(I).GT.T2(I)) THEN
          DUM=T1(I)
          T1(I)=T2(I)
          T2(I)=DUM
        ENDIF            
      ELSE
          T2(I)=1.0D30
          T1(I)=-1.0D30
      ENDIF
      ENDDO
      TIN1=MAX(T1(1),T1(2),T1(3))
      TOUT1=MIN(T2(1),T2(2),T2(3))
C segment size
      BSZ(1)=CR.FRAME.SIZE(1)/CR.NH-CR.DH
      BSZ(2)=CR.FRAME.SIZE(2)/CR.NV-CR.DV
      BSZ(3)=CR.FRAME.SIZE(3)/CR.NB-CR.DB
C Get entry and exit times for the segment itself      
      DO I=1,3      
      IF (ABS(K(I)).GT.1.0D-10) THEN
               T2(I)=(BSZ(I)/2.D0-R1(I))/K(I)
        T1(I)=(-BSZ(I)/2.D0-R1(I))/K(I)
        IF (T1(I).GT.T2(I)) THEN
          DUM=T1(I)
          T1(I)=T2(I)
          T2(I)=DUM
        ENDIF            
      ELSE
          T2(I)=1.0D30
          T1(I)=-1.0D30
      ENDIF
      ENDDO

c       if (R(1).LT.-8.5) then
c         write(*,1) 'R1: ',(R1(I),I=1,3),(BSZ(I),I=1,3)
c         write(*,1) 'T1: ',(T1(I),I=1,3)
c         write(*,1) 'T2: ',(T2(I),I=1,3)
c         pause
c       endif

      TIN2=MAX(T1(1),T1(2),T1(3))
      TOUT2=MIN(T2(1),T2(2),T2(3))

      END

C-----------------------------------------------------------------------------
      SUBROUTINE SEGSCATT(CR,J,DIR,R,K,PHI,Q,DT)
C Calculate scattering probability for J-th segment on the path along K(3)
C PHI is the vertical mosaic angle
C Q .. kinematical reflectivity (incl. DW) 
C R is the coordinate of the neutron in the assembly
C DT ... time-of-flight along K(3) through the segment 
C-----------------------------------------------------------------------------
      implicit none
      INCLUDE 'structures.inc'
      INCLUDE 'crystal.inc'
      RECORD /CRYSTAL/ CR
      REAL*8 SEC
      PARAMETER (SEC=4.85D-6)
      INTEGER*4 DIR,J,I,I1,I0(3),L
      REAL*8 DT,a,b,Z,Z1,Z2,sigma,KK,K0,GABS,Q,PHI,DALPHA
      REAL*8 KaG(3),K(3),G(3),R(3),R0(3),V(3)
      REAL*8 HMOS_DIST,ERF
C1     FORMAT(a10,10(1x,G12.6))
      
      IF (DT.LE.0) THEN
          PSEG0(J)=0.D0
          PSEG(J)=PSEG(J-1)
          RETURN
      ENDIF      
      
      KK=0.
      DO I=1,3
         KK=KK+K(I)**2
      END DO
      K0=SQRT(KK)
      
C get coordinates of the segment centre
      CALL SEGCOORD(CR,R,R0,I0)
      
      CALL LOCALG(CR,DIR,R,R0,I0,0.D0,PHI,G)
      
C get angular deviation from the Bragg condition (=alpha)            
      GABS=0.     
      DO I=1,3
         KaG(I)=K(I)+G(I)
         GABS=GABS+G(I)**2
      END DO
      GABS=SQRT(GABS)
      a=KaG(1)**2+ KaG(2)**2+KaG(3)**2-KK    ! (K+G)^2-K^2 
      b=DIR*GABS*(KaG(1)*CR.GAMA(1)+KaG(2)*CR.GAMA(2)+
     *       KaG(3)*CR.GAMA(3))   

      grad(J)=0.D0
      DO I=1,3
        IF (CR.MAPG(I)) THEN
          Z=0.D0
          DO I1=1,3
            Z=Z+DIR*CR.DG_DR(I,I1)*K(I1)
          ENDDO
          grad(J)=grad(J)+KaG(I)*Z
        ENDIF  
      END DO
C get gradient of angular deviation ALPHA            
      grad(J)=-grad(J)/b
C get angular deviation from the Bragg condition (=alpha)            
      alpha(J)=-a/(2.*b)
      
C      write(*,1) 'ALPHA: ',alpha(J)
C 2nd order correction for large bent crystals
      IF(abs(grad(J)).GE.1.D-6*K0*Q.AND.CR.HMOS.LE.SEC) THEN
      DALPHA=ALPHA(J)
      L=10 ! max. L iterations
      DO WHILE (ABS(DALPHA).GT.1D-7.AND.L.GT.0)
        L=L-1
        DO I=1,3
          V(I)=R(I)-alpha(J)/grad(J)*K(I)
        ENDDO
        CALL LOCALG(CR,DIR,V,R0,I0,0.D0,PHI,G)
        GABS=0.     
        DO I=1,3
          KaG(I)=K(I)+G(I)
          GABS=GABS+G(I)**2
        END DO
        GABS=SQRT(GABS)
        a=KaG(1)**2+ KaG(2)**2+KaG(3)**2-KK    ! (K+G)^2-K^2 
        DALPHA=-a/(2.*b)
        alpha(J)=alpha(J)+DALPHA 
C        write(*,1) 'DALPHA: ',-a/(2.*b)
      ENDDO
      ENDIF
C      pause


C// add random angle within Darwin box width
c        alpha(J)=alpha(J)+(RAN1(1)-0.5)*Q*CR.DEXT*1D-3/PI

C MOSAIC CRYSTAL
      IF(abs(grad(J)).LT.1.D-6*K0*Q.AND.CR.HMOS.GT.SEC) THEN
        sigma=K0*Q*HMOS_DIST(alpha(J)/CR.HMOS)/CR.HMOS*DT
      ELSE  
C MOSAIC CRYSTAL WITH G-GRADIENT
        IF(CR.HMOS.GT.SEC) THEN
           Z1=ERF((alpha(J)+grad(J)*DT)/CR.HMOS,0)
           Z2=ERF(alpha(J)/CR.HMOS,0)
           sigma=K0*Q*(Z1-Z2)/grad(J)
           IF(J.EQ.1.AND.ABS(ALPHA(J)).LT.SEC) sigma=0.D0 ! no back refl. in the same segment
C ONLY G-GRADIENT
        ELSE   
           Z1=-alpha(J)/grad(J)
           IF(Z1.GT.0.AND.Z1.LT.DT) THEN  
             sigma=K0*Q/abs(grad(J))
             IF(J.EQ.1.AND.ABS(ALPHA(J)).LT.SEC) sigma=0.D0 ! no back refl. in the same segment   
           ELSE
             sigma=0.D0
           ENDIF
        ENDIF                              
      ENDIF        
      IF(sigma.gt.14) then
          PSEG0(J)=1.D0
          PSEG(J)=1.D0
      else  
            PSEG0(J)=1.D0-exp(-sigma)
          PSEG(J)=1.D0-(1.D0-PSEG(J-1))*(1.D0-PSEG0(J))
      endif  
      
      END
      

C-------------------------------------------------------------------------
      SUBROUTINE SEGTRACE(CR,K0,Q,M,DT)
C return random step size to a point of reflection inside the M-th segment
C-------------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      INCLUDE 'crystal.inc'
      RECORD /CRYSTAL/ CR
      INTEGER*4 M
      REAL*8 SEC,ksi,Z,Z1,sigma,P0,DT,AA
      PARAMETER(SEC=4.85D-6)
      REAL*8 K0,Q
      REAL*4 RAN1
      REAL*8 ERF,HMOS_DIST
     
      DT=0.D0
      
C MOSAIC CRYSTAL
      IF(ABS(GRAD(M)).LT.1.D-6*K0*Q) THEN
          ksi=RAN1()
          P0=PSEG0(M)
          AA=ALPHA(M)/CR.HMOS
          sigma=K0*Q*HMOS_DIST(AA)/CR.HMOS
          DT=-Log(1-ksi*P0)/sigma
      ELSE
C MOSAIC CRYSTAL WITH D-GRADIENT
        IF(CR.HMOS.GT.SEC) THEN
          ksi=RAN1()
          P0=PSEG0(M)
          Z=GRAD(M)*Log(1-ksi*P0)/(K0*Q)
          AA=ALPHA(M)/CR.HMOS
C  ERF shoud be used only on (-inf.;0) because of num. precision
          IF(AA.GT.0) THEN
            Z1=ERF(-AA,0)
            IF(1.-Z1-Z.GT.0.5D0) THEN
              DT=-ERF(Z1+Z,1)-AA
            ELSE
              DT=ERF(1.-Z1-Z,1)-AA
            ENDIF
          ELSE
            Z1=ERF(AA,0)
            IF(Z1-Z.GT.0.5D0) THEN
              DT=-ERF(1.-Z1+Z,1)-AA
            ELSE
              DT=ERF(Z1-Z,1)-AA
            ENDIF             
          ENDIF
           DT=DT*CR.HMOS/GRAD(M)                     
C ONLY D-GRADIENT
        ELSE  
          DT=-ALPHA(M)/GRAD(M)
        ENDIF
      ENDIF
      
      END
      
C-----------------------------------------------------
      SUBROUTINE ERF_INIT(F,AMIN,AMAX)
C Calculate lookup tables for ERF function
C-----------------------------------------------------
      IMPLICIT NONE
      INTEGER*4 I,J
      REAL*8 SUM,Z1,Z2,Z3,A,B,DET,X1I,XJ,AMIN,AMAX
      INTEGER*4 DIM
      PARAMETER(DIM=1025)
      REAL*8 XMIN,DX,Y(DIM),XMIN1,DX1,Y1(DIM)
      COMMON /ERFCOM/ XMIN,DX,Y,XMIN1,DX1,Y1      
      REAL*8 F
      EXTERNAL F

C Generate cumulative function from F(X)
      SUM=0.
      DX=(AMAX-AMIN)/(DIM-1)
      XMIN=AMIN
      Y(1)=0.
      DO I=1,DIM-1
        Z1=XMIN+(I-1)*DX
        Z2=Z1+DX/2
        Z3=Z1+DX  
        SUM=SUM+F(Z1)+4*F(Z2)+F(Z3)
        Y(I+1)=SUM
      END DO  
      DO I=1,DIM
        Y(I)=Y(I)/Y(DIM)
      ENDDO
C Generate inverse cumulative function
      DX1=1.D+0/(DIM-1)
      XMIN1=Y(1)
      Y1(1)=XMIN
      Y1(DIM)=XMIN+(DIM-1)*DX
      J=1
      DO I=2,DIM-1
        X1I=XMIN1+(I-1)*DX1
        DO WHILE (Y(J).LT.X1I.AND.J.LT.DIM-1)
          J=J+1
        END DO
10      XJ=XMIN+(J-1)*DX
        A=(Y(J+1)+Y(J-1)-2*Y(J))/2
        B=(Y(J+1)-Y(J-1))/2
        IF (ABS(A).LT.1D-30) THEN
           J=J-1
           GOTO 10
        ELSE
          DET=B**2-4*A*(Y(J)-X1I)
          IF (DET.LE.0) then
            write(*,*) 'Error in ERF_INIT: ',DET,A,B
            pause
          ENDIF
          Z1=XJ+DX*(-B+SQRT(DET))/2/A
          Z2=XJ+DX*(-B-SQRT(DET))/2/A
          IF (ABS(Z2-XJ).LT.ABS(Z1-XJ)) Z1=Z2
          Y1(I)=Z1
        ENDIF
      END DO
  
      END

C----------------------------------------------------------------
      REAL*8 FUNCTION ERF(ARG,INV)
c Return cumulative function (or inverse, if INV=0)
c Uses lookup table generated by CUM_INIT
C----------------------------------------------------------------
      IMPLICIT NONE
      INTEGER*4 DIM,INV,J1,J2,J,I
      REAL*8 ARG,A,B,Z,XJ,DET,Z1,Z2,ARG1
      PARAMETER(DIM=1025)
      REAL*8 XMIN,DX,Y(DIM),XMIN1,DX1,Y1(DIM)
      REAL*8 ERF_INTERP
      COMMON /ERFCOM/ XMIN,DX,Y,XMIN1,DX1,Y1      

C Cumul. function
      IF (INV.NE.1) THEN  
        ERF=ERF_INTERP(XMIN,DX,Y,DIM,ARG)
C else Iverse cumul. function
      ELSE
        IF(ARG.LE.XMIN1) THEN
           ERF=Y1(1) ! left limit
        ELSE IF(ARG.GE.XMIN1+(DIM-1)*DX1) THEN
           ERF=Y1(DIM)  ! right limit
        ELSE
C Find J1,J2 so that Y(J1) > A >= Y(J2) and J2=J1+1
          ARG1=ARG
          IF(ARG.GT.9.D-1) ARG1=1.D0-ARG
          Z=(ARG1-XMIN1)/DX1+1
          I=INT(Z)
          Z1=(Y1(I)-XMIN)/DX+1
          Z2=(Y1(I+1)-XMIN)/DX+1   
          J1=INT(Z1)
          J2=INT(Z2)+1
          DO WHILE(J2.GT.J1+1)
            J=(J2+J1)/2
            IF(Y(J).LT.ARG1) THEN
               J1=J
            ELSE
               J2=J
            ENDIF
          ENDDO
C Set J so that Y(J) is close to ARG 
          J=J1
          IF(ARG1-Y(J1).GT.Y(J2)-ARG1) J=J2
C but avoid J=1 or J=DIM
          IF(J.LT.2) J=J+1
          IF(J.GT.DIM-1) J=J-1
C interpolate quadratically between Y(J-1) and Y(J+1) 
C return inverse value for Y=ARG
10        XJ=XMIN+(J-1)*DX
          A=(Y(J+1)+Y(J-1)-2*Y(J))/2
          B=(Y(J+1)-Y(J-1))/2
          IF(ABS(B).LT.1.D-30) THEN
             ERF=XJ
          ELSE IF (ABS(A).LT.1D-30) THEN
            ERF=XJ+Y(J+1)
            J=J-1
            GOTO 10
          ELSE
            DET=B**2-4*A*(Y(J)-ARG1)
            DET=SQRT(DET)
            Z1=XJ+DX*(-B+DET)/2/A
            Z2=XJ+DX*(-B-DET)/2/A
            IF (ABS(Z2-XJ).LT.ABS(Z1-XJ)) Z1=Z2
            IF(ARG.GT.9.D-1) Z1=-Z1
            ERF=Z1
          ENDIF
        ENDIF            
      ENDIF
      END 

C-------------------------------------------------------------------
      REAL*8 FUNCTION ERF_INTERP(XMIN,DX,Y,DIM,A)
C Quadratic interpolation in Y array with equidistant X
C given by XMIN,DX. MIN,MAX are values to be returned outside limits
C-------------------------------------------------------------------
      IMPLICIT NONE
      INTEGER*4 DIM,I
      REAL*8 XMIN,DX,A,Z,XMAX
      REAL*8 Y(DIM)   

      XMAX=XMIN+(DIM-1)*DX
      if (A.LE.XMIN) THEN 
         ERF_INTERP=Y(1)
      elseif (A.GE.XMAX) THEN         
         ERF_INTERP=Y(DIM)
      elseif (A.LE.XMIN+DX) THEN         
         ERF_INTERP=Y(1)+(Y(2)-Y(2))*(A-XMIN)/DX
      elseif (A.GE.XMAX-DX) THEN         
         ERF_INTERP=Y(DIM-1)+(Y(DIM)-Y(DIM-1))*(A-XMAX+DX)/DX
      else
        Z=(A-XMIN)/DX+1
        I=NINT(Z)
        if (I.EQ.DIM-1) I=I-1
        IF (I.EQ.2) I=3
        ERF_INTERP=Y(I)+(Y(I+1)-Y(I-1))/2*(Z-I)+
     *             (Y(I+1)+Y(I-1)-2*Y(I))/2*(Z-I)**2
      endif
      end

      
C        --------------------------------------------------
        LOGICAL*4 FUNCTION CR_INSIDE(CRYST,R)
C       INSIDE function for CRSYTAL object ... takes into
C       account curved surface of a bent crystal plate.        
C     NOT USED IN THE CURRENT VERSION !
C        --------------------------------------------------        
        implicit none

        INCLUDE 'structures.inc'
      
        RECORD /CRYSTAL/ CRYST
        REAL*8 R(3),R0(3)
        LOGICAL*4 INSIDE
        
        
        R0(3)=R(3)-R(1)**2*CRYST.RH-R(2)**2*CRYST.RV
        R0(1)=R(1)
        R0(2)=R(2)
        CR_INSIDE=INSIDE(CRYST.FRAME,R0)
        
        RETURN
        END
        
