C//////////////////////////////////////////////////////////////////////
C////                                                              //// 
C////  NEutron Scattering Simulation - v.2.0, (c) J.Saroun, 2004   ////
C////                                                              //// 
C//////////////////////////////////////////////////////////////////////
C////
C////  Subroutines describing objects - PARABOLIC GUIDE
C////  
C////                          
C//////////////////////////////////////////////////////////////////////

C---------------------------------------------------------------------
      SUBROUTINE PGUIDE_INIT(OBJ)
C----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      
      REAL*4 PGUIDE_A
      REAL*4 XX
      REAL*8 f,ZL,DIF,A,H1,H2,V1,V2
      RECORD /BENDER/ OBJ
      INTEGER*4 I
      
      CALL SLIT_INIT(OBJ.FRAME)

C limit for number of slits = 127
      IF (OBJ.NLH.GT.127) OBJ.NLH=127
      IF (OBJ.NLV.GT.127) OBJ.NLV=127
      
C HORIZONTAL      
c parabolic profile is determined by the dimensions
      H1=OBJ.FRAME.SIZE(1)
      H2=OBJ.W2
      IF (H2.EQ.H1.OR.OBJ.CH.EQ.0) THEN  !  no parabola, flat walls
         OBJ.CH=0.D0
      ELSE IF (H2.GT.H1) THEN
	 DIF=H2**2-H1**2
	 A=DIF/4.D0/OBJ.FRAME.SIZE(3)
	 OBJ.CH=A-H1**2*OBJ.FRAME.SIZE(3)/DIF
      ELSE IF (H2.LT.H1) THEN
	 DIF=H1**2-H2**2
	 A=DIF/4.D0/OBJ.FRAME.SIZE(3)
	 OBJ.CH=H2**2*OBJ.FRAME.SIZE(3)/DIF-A
      ENDIF  

      f=OBJ.CH
      IF (f.EQ.0) THEN ! flat lamellas, AH & LH are angles & positions, respectively
        DO I=0,OBJ.NLH
           ZL=I*1.D0/OBJ.NLH - 0.5D0
	   OBJ.AH(I)=ZL*(OBJ.W2-OBJ.FRAME.SIZE(1))/OBJ.FRAME.SIZE(3)
	   OBJ.LH(I)=ZL*OBJ.FRAME.SIZE(1)
        ENDDO
      ELSE  ! parabolic lamellas, AH & LH are parameters & lengths, respectively
	IF (f.GT.0) f=f+OBJ.FRAME.SIZE(3)
        DO I=0,OBJ.NLH
	     OBJ.AH(I)=PGUIDE_A(I,OBJ.NLH,OBJ.FRAME.SIZE(1),f)
        ENDDO
        IF (OBJ.TYP.EQ.3.AND.OBJ.NLH.GT.1) THEN  ! set optimum lengths of lamellae
          DO I=0,OBJ.NLH
	    XX=ABS(I-0.5*OBJ.NLH)
	    IF (XX.EQ.0) THEN
	       OBJ.LH(I)=OBJ.FRAME.SIZE(3)
	    ELSE
	       OBJ.LH(I)=MIN(2.0*ABS(OBJ.CH)/XX,OBJ.FRAME.SIZE(3))
	    ENDIF   
c      write(*,*) 'lam: ',I,'  ',OBJ.LH(I),'  ',
c     &      XX*OBJ.FRAME.SIZE(1)/OBJ.NLH
          ENDDO
	ELSE
          DO I=0,OBJ.NLH
	     OBJ.LH(I)=OBJ.FRAME.SIZE(3)
          ENDDO	
	ENDIF  
      ENDIF
      
C VERTICAL      
c parabolic profile is determined by the dimensions
      V1=OBJ.FRAME.SIZE(2)
      V2=OBJ.H2
      IF (V2.EQ.V1.OR.OBJ.CV.EQ.0) THEN  !  no parabola, flat walls
	 OBJ.CV=0.D0
      ELSE IF (V2.GT.V1) THEN
	 DIF=V2**2-V1**2
	 A=DIF/4.D0/OBJ.FRAME.SIZE(3)
	 OBJ.CV=A-V1**2*OBJ.FRAME.SIZE(3)/DIF
      ELSE IF (V2.LT.V1) THEN
	 DIF=V1**2-V2**2
	 A=DIF/4.D0/OBJ.FRAME.SIZE(3)
	 OBJ.CV=V2**2*OBJ.FRAME.SIZE(3)/DIF-A
      ENDIF  	
      f=OBJ.CV
      IF (f.EQ.0) THEN ! flat lamellas, AH & LH are angles & positions, respectively
        DO I=0,OBJ.NLV
           ZL=I*1.D0/OBJ.NLV - 0.5D0
	   OBJ.AV(I)=ZL*(OBJ.H2-OBJ.FRAME.SIZE(2))/OBJ.FRAME.SIZE(3)
	   OBJ.LV(I)=ZL*OBJ.FRAME.SIZE(2)
        ENDDO
      ELSE  ! parabolic lamellas, AH & LH are parameters & lengths, respectively
	IF (f.GT.0) f=f+OBJ.FRAME.SIZE(3)
        DO I=0,OBJ.NLV
	     OBJ.AV(I)=PGUIDE_A(I,OBJ.NLV,OBJ.FRAME.SIZE(2),f)
        ENDDO
        IF (OBJ.TYP.EQ.3.AND.OBJ.NLV.GT.1) THEN  ! set optimum lengths of lamellae
          DO I=0,OBJ.NLV
	    XX=ABS(I-0.5*OBJ.NLV)
	    IF (XX.EQ.0) THEN
	       OBJ.LV(I)=OBJ.FRAME.SIZE(3)
	    ELSE
	       OBJ.LV(I)=MIN(2.0*ABS(OBJ.CV)/XX,OBJ.FRAME.SIZE(3))
	    ENDIF   
          ENDDO
	ELSE
          DO I=0,OBJ.NLV
	     OBJ.LV(I)=OBJ.FRAME.SIZE(3)
          ENDDO	
	ENDIF  
      ENDIF
      
      END

C---------------------------------------------------------------------
      REAL*8 FUNCTION PGUIDE_LAM(OBJ,ID,IL,Z,IOR)
C function describing lamella profile
C ID   ... derivative
C IL   ... lamella index 
C Z    ... distance from guide entry
C IOR  ... horizontal (1) or vertical (2) 
C----------------------------------------------------------------------     
      IMPLICIT NONE
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      
      REAL*8 a,f,zz,aa,x0
      REAL*8 Z
      INTEGER*4 ID,IL,IOR          
      
      IF (IOR.EQ.2) THEN   ! vertical slit
        f=OBJ.CV
        a=OBJ.AV(IL)
        x0=OBJ.LV(IL)
      ELSE                 ! horizontal slit
        f=OBJ.CH
        a=OBJ.AH(IL)
        x0=OBJ.LH(IL)
      ENDIF
      
      IF (f.LT.0) THEN     ! focal point before the guide
        zz=z-f+abs(a)	      
      ELSE IF (f.GT.0) THEN   ! focal point behind the guide
        zz=-z+f+OBJ.FRAME.SIZE(3)+abs(a)
      ELSE
        zz=0
      ENDIF
      aa=SIGN(1.D0,a)*SQRT(abs(a))
            
      IF (zz.LE.0.D0.AND.f.NE.0) THEN
        PGUIDE_LAM=0.D0
	RETURN
      ENDIF       

c zero deriv.      
      IF (ID.LE.0) THEN
        IF (f.EQ.0) THEN
	  PGUIDE_LAM=x0+a*z
	ELSE
          PGUIDE_LAM=2.D0*aa*SQRT(zz)
	ENDIF  
c 1st deriv.      
      ELSE IF (ID.EQ.1) THEN
        IF (f.EQ.0) THEN
	  PGUIDE_LAM=a
	ELSE
          PGUIDE_LAM=-aa*SIGN(1.D0,f)/SQRT(zz)
	ENDIF  
c 2nd deriv.      
      ELSE IF (ID.EQ.2) THEN
        IF (f.EQ.0) THEN
	  PGUIDE_LAM=0.D0
	ELSE
          PGUIDE_LAM=-aa/SQRT(zz)/zz/2.D0
	ENDIF  
      ELSE
        PGUIDE_LAM=0.D0
      ENDIF
      
      END

      
C-------------------------------------------------------------------------------
      REAL*4 FUNCTION PGUIDE_A(il,nl,w,f)
C parabola parameter for il-th lamella
C il ... lamella index, counted from right, right side il=0
C f  ... focal distance
C w  ... width of the entry 
C nl ... number of slits (number of lamellae + 1)
C sign(a) determines which side from the guide center: right/bottom(<0) or left/top(>0) 
C--------------------------------------------------------------------------------
      IMPLICIT NONE
      REAL*8 f,w,z
      REAL*4 xx
      INTEGER*4 il,nl
            
      xx=w*(il*1.E0/nl-0.5D0)
      IF (abs(xx).LE.1.E-10) THEN
         PGUIDE_A=0.D0
      ELSE
         PGUIDE_A=SIGN(1.E0,xx)*(SQRT(f**2+xx**2)-abs(f))/2.E0
      ENDIF
      Z=SIGN(1.E0,xx)*(SQRT(f**2+xx**2)-abs(f))/2.E0
      
      END
      

C----------------------------------------------------------------------------
      REAL*8 FUNCTION PGUIDE_CROSS(kx,kz,x,z,a,f,lmax)
C return path length along z-coordinate to a cross-point with parabolic surface 
C if there are 2 solutions, take the minimum one which is > 0
C return 10^30 if there is no solution 0 < result < lmax
C a      .. parabola parameter
C f      .. focal distance from guide entry (origin of z coordinate) 
C lmax   .. limit distance (lamella length)
C x,z    .. starting point (z // guide axis)
C kx,kz  .. ray direction
C kz is assumed positive and >> kx
C----------------------------------------------------------------------------
      IMPLICIT NONE
      REAL*8 EPS
      PARAMETER (EPS=1.D-7)
      REAL*8 kx,kz,x,z,a,f,lmax     
      REAL*8 LZ,Y1,Y2,tang,s,aa,dtm

c10    FORMAT(a10,1x,8(1x,G13.7))
c      write(*,10) 'cross',kx,kz,x,z,a,f,lmax
            
      IF (ABS(kx).LT.1.D-6*kz) THEN  ! beam along z
        IF (a.GT.EPS) THEN
          LZ=f+SIGN(1.D0,f)*(a-x**2/a/4.D0)-z
	ELSE
	  LZ=1.D30
	ENDIF
      ELSE 
        tang=kz/kx
        dtm=a*(1+tang**2) + SIGN(1.D0,f)*(x*tang+f-z)
	IF (dtm.LT.0.D0) THEN
	   LZ=1.D30
	ELSE   
           s=2*SIGN(1.D0,f)*SIGN(1.D0,kx)
           aa=SQRT(a)	
           Y1=(s*aa*(-aa*abs(tang)+SQRT(dtm))-X)*tang
           Y2=(s*aa*(-aa*abs(tang)-SQRT(dtm))-X)*tang
c      write(*,10) 'cross',Y1,Y2,dtm
           IF(Y1.LE.EPS.AND.Y2.LE.EPS) THEN
	      LZ=1.D30
           ELSE 
              IF (Y1.GT.EPS.AND.Y2.GT.EPS) THEN
                 LZ=MIN(Y1,Y2)
              ELSE 
                 LZ=MAX(Y1,Y2)
              ENDIF             
           ENDIF   
	ENDIF  
      ENDIF
c      write(*,10) 'cross',Y1,Y2,LZ,lmax
c      pause
      IF (LZ.GT.lmax) LZ=1.D30
      PGUIDE_CROSS=LZ
            
      END


C---------------------------------------------------------------------
      SUBROUTINE PGUIDE_CON(OBJ,R,K,IH,IV,IC,DT,Q)
C find next contact with a slit side
C moves to the next contact point, or to the exit
C turns K, i.e. returns new K=K+Q !
C *** INPUT:
C IH,IV   ... slit indices
C R(3)    ... neutron coordinate
C K(3)    ... neutron k-vector
C *** RETURN:
C Q     .. reflection vector (magnitude)
C DT    .. time to reach the next contact point
C IC    .. which wall,  right(1), left(2), bottom (3) or top(4)
C IC=0  .. no contact, pass through
C----------------------------------------------------------------------     
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      RECORD /BENDER/ OBJ
      REAL*8 R(3),K(3),Q,DT
      INTEGER*4 IH,IV,IC
      REAL*8 PGUIDE_CROSS,GUIDE_CROSS,PGUIDE_LAM
      REAL*8 kx,kz,x,z,a,f,lmax
      REAL*8 LZ(4),ANG,T0,N(3)
      INTEGER*4 I
      REAL*8 OSCD,OSCA
      COMMON /OSCBEND/ OSCD,OSCA
 
c10    FORMAT(a11,1x,6(1x,G11.5),2(1x,I4),2(1x,G10.4))
c      write(*,10) 'CON START: ',R,K,IH,IV,OBJ.CH

      z=R(3)
      kz=K(3)
      
      T0=OBJ.FRAME.SIZE(3)-z
C HORIZONTAL RIGHT:
      f=OBJ.CH
      kx=K(1)
      lmax=T0
      IF (f.EQ.0) THEN
         a=OBJ.AH(IH)-OSCA
         x=R(1)-OBJ.DLH/2.D0-OBJ.LH(IH)-OSCD-R(3)*OSCA
         LZ(2)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=ABS(OBJ.AH(IH))
         x=R(1)-OBJ.DLH/2.D0
	 if (f.GT.0) f=f+OBJ.FRAME.SIZE(3)               
         if (f.LT.0) lmax=OBJ.LH(IH)-z
	 LZ(2)=PGUIDE_CROSS(kx,kz,x,z,a,f,lmax)
      ENDIF
C HORIZONTAL LEFT:
      IF (f.EQ.0) THEN
         a=OBJ.AH(IH+1)-OSCA
         x=R(1)+OBJ.DLH/2.D0-OBJ.LH(IH+1)-OSCD-R(3)*OSCA
         LZ(1)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=ABS(OBJ.AH(IH+1))
         x=R(1)+OBJ.DLH/2.D0
         if (f.LT.0) lmax=OBJ.LH(IH+1)-z
         LZ(1)=PGUIDE_CROSS(kx,kz,x,z,a,f,lmax)
      ENDIF
C VERTICAL BOTTOM:
      f=OBJ.CV
      kx=K(2)
      lmax=T0
      IF (f.EQ.0) THEN
         a=OBJ.AV(IV)
         x=R(2)-OBJ.DLV/2.D0-OBJ.LV(IV)
         LZ(4)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=ABS(OBJ.AV(IV))
         x=R(2)-OBJ.DLV/2.D0
         if (f.GT.0) f=f+OBJ.FRAME.SIZE(3)      
         if (f.LT.0) lmax=OBJ.LV(IV)-z
         LZ(4)=PGUIDE_CROSS(kx,kz,x,z,a,f,lmax)
      ENDIF
C VERTICAL TOP:
      IF (f.EQ.0) THEN
         a=OBJ.AV(IV+1)
         x=R(2)+OBJ.DLV/2.D0-OBJ.LV(IV+1)
         LZ(3)=GUIDE_CROSS(kx,kz,x,z,a,f)
      ELSE
         a=ABS(OBJ.AV(IV+1))
         x=R(2)+OBJ.DLV/2.D0
         if (f.LT.0) lmax=OBJ.LV(IV+1)-z
         LZ(3)=PGUIDE_CROSS(kx,kz,x,z,a,f,lmax)
      ENDIF
      
c      write(*,10) 'kz*times: ',LZ(1),LZ(2)
      
      DT=MIN(LZ(1),LZ(2),LZ(3),LZ(4),T0)
      
      IC=0
      DO I=1,4
        IF (DT.EQ.LZ(I)) IC=I
      ENDDO 
      IF (IC.EQ.0) THEN ! no contact, passed through
        DT=T0
	Q=0.D0
	GOTO 50 
      ENDIF

C get the surface normal vector
      DO I=1,3
        N(I)=0.D0
      ENDDO	 
      IF (IC.EQ.2) THEN
        ANG=PGUIDE_LAM(OBJ,1,IH,Z+DT,1)
	N(3)=-ANG/SQRT((1.D0+ANG**2))
	N(1)=SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.1) THEN
        ANG=PGUIDE_LAM(OBJ,1,IH+1,Z+DT,1)
	N(3)=ANG/SQRT((1.D0+ANG**2))
	N(1)=-SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.4) THEN
        ANG=PGUIDE_LAM(OBJ,1,IV,Z+DT,2)
	N(3)=-ANG/SQRT((1.D0+ANG**2))
	N(2)=SQRT(1.D0-N(3)**2)
      ELSE IF (IC.EQ.3) THEN
        ANG=PGUIDE_LAM(OBJ,1,IV+1,Z+DT,2)    
	N(3)=ANG/SQRT((1.D0+ANG**2))
	N(2)=-SQRT(1.D0-N(3)**2)
      ENDIF 

c      write(*,10) 'angles: ',ANG,N
      
C scattering vector:
      Q=0.D0
      DO I=1,3
        Q=Q-K(I)*N(I)
      ENDDO	 
C move to the point of reflection:
50    DO I=1,3
        R(I)=R(I)+K(I)/K(3)*DT
      ENDDO   
C turn K-vector:
      DO I=1,3
        K(I)=K(I)+2.D0*Q*N(I)
      ENDDO   
C convert T to time units	
      DT=DT/kz/HOVM
      
c      write(*,10) 'lamellae: ',
c     & PGUIDE_LAM(OBJ,0,IH,R(3),1)+OBJ.DLH/2.D0,
c     & PGUIDE_LAM(OBJ,0,IH+1,R(3),1)-OBJ.DLH/2.D0
c      write(*,10) ' CON HIT: ',R,K,IC,OBJ.FRAME.COUNT,DT,Q
c      pause
      
      END
      
C----------------------------------------------------
      SUBROUTINE PGUIDE_GO(OBJ,R,K,P,T,S)
C GO procedure for parabolic guide (TYP=2)      
C INPUT:   assume R,K at the entry in local coordinates, i.e. R(3)=0 !
C RETURN:  R,K at the exit in local coordinates, P=P*transmission, T=T+passage time
C----------------------------------------------------	
      IMPLICIT NONE

      INCLUDE 'structures.inc'
      
      REAL*8 EPS
      PARAMETER (EPS=1.D-7)
      RECORD /BENDER/ OBJ
      REAL*8 R(3),K(3),P,T,S
      LOGICAL*4 BENDER_PASS
      REAL*8 BENDER_REF      
      INTEGER*4 IH,IV,IC,I
      REAL*8 DUM,KK,PP,TT,DT,Q
      REAL*8 lx,xx,fx,sx,sk,LD,Z,DETM,DTH,DTV
      
c      LOGICAL*4 rep
c      rep=(OBJ.FRAME.NAME(1:5).EQ.'guide'.AND.OBJ.FRAME.count.GT.10000)
c      rep=(rep.AND..NOT.OBJ.FRAME.NAME(1:6).EQ.'guide_')
c10    FORMAT(a10,1x,6(1x,G13.7),3(1x,I4),2(1x,G10.4))
c11    FORMAT(a10,1x,8(1x,G13.7))
c      if (rep) write(*,10) 'guide entry ',OBJ.CH,R
      
      IF (OBJ.TYP.EQ.3.AND.(OBJ.CH.GT.0.OR.OBJ.CV.GT.0)) THEN
        LD=OBJ.FRAME.SIZE(3)-R(3) ! distance to the guide exit        
C try horizontal
        IF (OBJ.CH.GT.0.AND.OBJ.NLH.GT.2) THEN ! focus is behind exit
          xx=R(1)
	  sx=sign(1.D0,xx)
	  sk=sign(1.D0,K(1))
	  fx=2.D0*OBJ.CH*OBJ.FRAME.SIZE(1)/OBJ.NLH
	  lx=OBJ.FRAME.SIZE(3)
	  IF(ABS(xx).GT.EPS) lx=MIN(fx/ABS(xx),lx) ! "optimum" lamella length
c      if (rep) write(*,10) 'lam1 ',xx,fx,lx
          IF (lx.LT.LD) THEN
	    Z=LD/K(3)-xx/K(1)
	    DETM=Z**2+4.D0*(LD*xx-fx*sx)/K(1)/K(3)
            DTH=(Z+sx*sk*SQRT(DETM))/2.D0   ! positive solution of quadratic equation
	  ELSE
	    DTH=0.D0
	  ENDIF
	ELSE
	  DTH=0.D0
	ENDIF
C try vertical
        IF (OBJ.CV.GT.0.AND.OBJ.NLV.GT.2) THEN ! focus is behind exit
          xx=R(2)
	  sx=sign(1.D0,xx)
	  sk=sign(1.D0,K(2))
	  fx=2.D0*OBJ.CV*OBJ.FRAME.SIZE(2)/OBJ.NLV
	  lx=OBJ.FRAME.SIZE(3)
	  IF(ABS(xx).GT.EPS) lx=MIN(fx/ABS(xx),lx) ! "optimum" lamella length
          IF (lx.LT.LD) THEN
	    Z=LD/K(3)-xx/K(2)
	    DETM=Z**2 + 4.D0*(LD*xx-fx*sx)/K(2)/K(3)
            DTV=(Z+sx*sk*SQRT(DETM))/2.D0   ! positive solution of quadratic equation
	  ELSE
	    DTV=0.D0
	  ENDIF
	ELSE
	  DTV=0.D0
	ENDIF
	DT=MIN(DTH,DTV)
C move to the real guide entry	
	IF (DT.GT.0) THEN
	  DO I=1,3
	    R(I)=R(I)+K(I)*DT
	  ENDDO
	  T=T+DT
	ENDIF
      ENDIF
c      if (rep) write(*,10) 'lam2 ',DTH,DTV,DT
                   
c      if (rep) write(*,10) 'start',R,K
C  check passage through the entry
      IF (.NOT.BENDER_PASS(OBJ,R,IH,IV)) GOTO 100                  

      KK=SQRT(K(1)**2+K(2)**2+K(3)**2)
      PP=1.D0
      TT=0.D0
c      I=0
c      if (rep) write(*,10) 'start',R,K

C  iterate through reflections
1     CALL PGUIDE_CON(OBJ,R,K,IH,IV,IC,DT,Q)
c      if (rep) write(*,10) 'bounce',R,K,IH,IV,IC,DT,Q
      TT=TT+DT
c debug: prevent infinite loops      
c      I=I+1
c      IF (I.GT.100000) THEN
c	 write(*,*) 'too many iterations through parabolic guide, i=',I
c         stop
c      ENDIF      
      IF (IC.GT.0) THEN
         PP=PP*BENDER_REF(IC-1,OBJ,Q,S)
	 IF (PP.LT.1.D-4) GOTO 100
	 GOTO 1
      ENDIF

c      if (rep) write(*,11) 'OK',R,K,PP,TT
c      if (rep) pause
     
C renormalize K due to numerical errors
      DUM=SQRT(K(1)**2+K(2)**2+K(3)**2)
      DO I=1,3
        K(I)=K(I)*KK/DUM
      ENDDO
      P=P*PP
      T=T+TT
      RETURN

100   CONTINUE
c      if (rep) write(*,10) 'failed',R,K,IH,IV
c      if (rep) pause
      P=0.D0
 	
      END              
      
      
      
      
