C////////////////////////////////////////////////////////////////////////
C
C                 ************************************ 
C                 ***                              *** 
C                 ***        S I M R E S           *** 
C                 ***            (PWD)             ***
C                 ***   (C) J.Saroun & J.Kulda     *** 
C                 ***        ILL Grenoble          ***
C                 ***      evaluation version      ***
C                 ************************************ 
C
C A clone of RESTRAX: http://omega.ujf.cas.cz/restrax
C
C Provides more flexible (and more realistic) ray-tracing code useful for simulation 
C of newly designed or upgraded instruments and optimisation of their configuration. 
C This version permits to simulate intensity and distribution of neutron beam in both 
C real and momentum subspaces at different positions along the TAS beamline. 
C Arrangements derived from TAS setup can also be simulated - they involve e.g. powder
C diffractometers equipped with multidetectors, neutron guides or different configurations 
C of primary spectormeter (i.e. crystal monochromator with series of collimator or 
C guide segments).
C 
C****************************************************************************
C *** For all additional information contact the authors:                 ***
C ***                                                                     ***
C ***                 kulda@ill.fr        saroun@ujf.cas.cz               ***
C ***                                                                     ***
C****************************************************************************

C***********************************************************
C
C ONLY M.C. SIMULATION OF NEUTRON FLUX IN THIS VERSION !!!
C
C***********************************************************
C-------------------------------------
      SUBROUTINE RESTRAX_MAIN
C Main unit for console application
C-------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'      
      INCLUDE 'inout.inc'
      INCLUDE 'linp.inc'
      CHARACTER*(128) LINE
      CHARACTER*1 CH
      INTEGER*4 L
1     FORMAT(a)
2     FORMAT(a,$)
      
      CALL CMD_HANDLE('SETLINP')      
      DO WHILE (.TRUE.)
10      IF (linp_in.EQ.5) WRITE(linp_out,2) linp_p(1:linp_np)//'> ' 
        IF (linp_eof.GT.0) GOTO 20
        READ(SINP,1,END=20) LINE ! treat EOF
        CH=LINE(1:1)
        IF (CH.EQ.'#'.OR.CH.EQ.' '.OR.CH.EQ.char(0)) GOTO 10
        L=LEN(LINE)
        CALL CMD_HANDLE(LINE(1:L))
        GOTO 10        
20      CALL REINP(' ')
        CALL REOUT(' ') ! end of job file -> close also output file
        CALL LINPSETIO(SINP,SOUT,SMES)
      ENDDO
      
      END

C-------------------------------------------------------------
      SUBROUTINE BEFORE(IERR)
C  Call whenever the setup may have changed
C  Updates calculated auxilliary fields   
C  eturns IRES=0, if everything is OK 
C------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INCLUDE 'rescal.inc'
      INCLUDE 'trax.inc'
      INTEGER*4 IER,IERR
      COMMON /ERROR/IER
      REAL*8 Z,Z1,CO,SI
C look on the commons for collimators at a more convenient scope
      INTEGER*4 NCOL(4),I,J
      REAL*8 VLC(6,4)
      EQUIVALENCE (NCOL(1),NFM)
      EQUIVALENCE (VLC(1,1),VLCANM)
      
C ATTENTION : order of following routines may be important !!!
      IER=0

C///  scattering triangle in STP record
      STP.NFX=RES_DAT(i_FX)
      STP.SM=RES_DAT(i_SM)
      STP.SS=RES_DAT(i_SS)
      STP.SA=RES_DAT(i_SA)
      STP.KFIX=RES_DAT(i_KFIX)
      IF (STP.NFX.EQ.1.) THEN
         STP.EI0=HSQOV2M*STP.KFIX**2
         STP.EF0=STP.EI0-RES_DAT(i_EN)
      ELSE
         STP.EF0=HSQOV2M*STP.KFIX**2
         STP.EI0=STP.EF0+RES_DAT(i_EN)
      END IF                                      
      STP.KI=SQRT(STP.EI0/HSQOV2M)
      STP.KF=SQRT(STP.EF0/HSQOV2M)

      call QNORM(QHKL,Z,Z1)
      STP.Q=Z1
      STP.E=HSQOV2M*(STP.KI**2-STP.KF**2)
      
C// scattering angle 
      IF (STP.Q.EQ.0) THEN
        COMEGA=1
        SOMEGA=0
        OMEGA=0
      ELSE        
        COMEGA=-(STP.Q**2-STP.KI**2-STP.KF**2)/(2*STP.KI*STP.KF)
        IF(ABS(COMEGA).GT.1) GOTO 999
        SOMEGA=SIGN(1,STP.SS)*SQRT(1-COMEGA**2)
        OMEGA=ASIN(SOMEGA)
        IF (COMEGA.LT.0) OMEGA=SIGN(1,STP.SS)*PI-OMEGA
      ENDIF

C///  trans. matrix CN->lab
      DO 60 I=1,3
      DO 60 J=1,3
         MLC(I,J)=0.
60    CONTINUE
      IF (STP.Q.EQ.0) THEN
        CO=1
        SI=0
      ELSE              
        CO=(STP.KF**2-STP.KI**2-STP.Q**2)/(2*STP.KI*STP.Q)
        IF(ABS(CO).GT.1) GOTO 999
        SI=SIGN(1,STP.SS)*SQRT(1-CO**2)
      ENDIF  
      MLC(1,1)=SI
      MLC(1,2)=CO
      MLC(2,3)=1.
      MLC(3,1)=CO
      MLC(3,2)=-SI

C// transformation matrices
      CALL RECLAT          !   compute reciprocal lattice parameters and matrices
      CALL TRANSMAT        !   create transformation matrices for coordinate systems
      
C  collimator parameters from TRAX
c ////   if ALPHA(I)<500  then the coarse collimator is ignored
c ////   if ALPHA(I)>=500 then the Soller collimator is ignored 
c ////   if ALPHA(I)=0 then no collimation is considered

      DO I=1,4
        ALPHA(I)=RES_DAT(i_ALF1+I-1)
        BETA(I)=RES_DAT(i_BET1+I-1)
        NCOL(I)=-1
        IF (ALPHA(I).GE.500.AND.VLC(2,I).NE.0) THEN
          ALPHA(I)=0.
          NCOL(I)=1
        ENDIF
      ENDDO 

      IERR=IER
      RETURN
 
      
999   IERR=1              
      END                                 
                         
C-----------------------------
      SUBROUTINE LOGO
C----------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'config.inc'

1     FORMAT(2x,'-----------------------------------------------',/,
     &       2x,'S I M R E S  - Monte Carlo ray-tracing ',/,
     &       2x,'Version: ',a40,/,
     &       2x,'Build:   ',a40,/,
     &       2x,'-----------------------------------------------',/,
     &       2x,'(C) J.Saroun & J.Kulda',/,
     &       2x,'ILL Grenoble, NPI Rez near Prague',/,
     &       2x,' ',/,
     &       2x,'-----------------------------------------------',/,
     &       2x,'type ? for command list',/,/)
     

      write(SOUT,1) PACKAGE_VERSION,PACKAGE_DATE  
      return
      end

C***********************************************************************
        SUBROUTINE UNITS(SARG)
C***********************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      CHARACTER*(*) SARG
      CHARACTER*1 CH

1     FORMAT(' Energy In m[eV] Or T[Hz]? (meV) ',$)
2     FORMAT('THz')
5     FORMAT('meV')
3     FORMAT(A)
4     FORMAT(' Units are ',A)
        
      CH=SARG(1:1)
      IF(CH.EQ.' ') THEN
        write(smes,1)
        read(sinp,3) CH
        CALL MKUPCASE(CH)
      ENDIF  
      IF(CH.EQ.'T') THEN
          EUNI=0.24181
          WRITE(CUNIT,2)
      ELSE
          EUNI=1.
          WRITE(CUNIT,5)
      ENDIF
      write(sout,4) CUNIT
      END
C
C

C-----------------------------------------------------------        
      SUBROUTINE GETROANAL(RO)
C return "optimal" monochromator and analyzer curvatures 
C calculated analytically
C *** J.S. 3/6/1997     
C-----------------------------------------------------------      

      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INCLUDE 'trax.inc'
      INCLUDE 'rescal.inc'

      REAL*4 RO(4)
      REAL*8 THM,CHIM,THA,CHIA
 
      THM=PI/SQRT(STP.EI0/HSQOV2M)/RES_DAT(i_DM)
      THM=ABS(ASIN(THM))
      CHIM=HIMON*deg      
      THA=PI/SQRT(STP.EF0/HSQOV2M)/RES_DAT(i_DA)
      THA=ABS(ASIN(THA))
      CHIA=HIANA*deg
c      RO(1)=SIN(THM+CHIM)/2./VL1*100
c change to monochromatic focusing:
      RO(1)=SIN(THM+CHIM)/VL1*100
c      write(*,*) 'VL1, THM, CHIM: ',VL1,THM*180/PI,CHIM*180/PI
      RO(2)=1./VL1/(2.*SIN(THM)*COS(CHIM))*100
c      RO(3)=(VL2*SIN(THA+CHIA) + VL3*SIN(THA-CHIA))/2./VL2/VL3*100
      RO(3)=SIN(THA-CHIA)/VL2*100
      RO(4)=(1./VL2+1./VL3)/(2.*SIN(THA)*COS(CHIA))*100
      END

          
C-----------------------------------------------------------        
      SUBROUTINE GETRO
C generates "optimal" monochromator and analyzer curvatures      
C-----------------------------------------------------------                    
      IMPLICIT NONE
      INCLUDE 'inout.inc'
      include 'rescal.inc'

      REAL*4 RO(4)
      INTEGER*4 I
      CHARACTER*10 REMARK(4)
            
1     FORMAT(1x,a4,' = ',F8.4,' [m-1] ',a10)
                 
      CALL GETROANAL(RO)
      DO I=1,4
        REMARK(I)=' '
        IF((NOS.EQ.0).OR.((NOS.GE.I).AND.(RET(I).EQ.1.))) THEN
           RES_DAT(i_ROMH+I-1)=RO(I)
           REMARK(I)=' changed'
        ENDIF
        WRITE(SOUT,1) RES_NAM(i_ROMH+I-1),RO(I),REMARK(I)   
      ENDDO
            
      END


C-----------------------------------------------------------        
      SUBROUTINE GETROOPTMC
C Optimize curvature with M.C. simulation
C Only one of the curvatures can be optimized  
C *** J.S. 5/7/2001     
C-----------------------------------------------------------      
              
      IMPLICIT NONE
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'

      REAL*4 RO(4)
      INTEGER*4 IERR,I
      COMMON /ERROR/IERR
      INTEGER*4 OPTPAR,OPTMERIT
      REAL*8 OPTEV
      COMMON /MCOPTIM/ OPTPAR,OPTMERIT,OPTEV
      REAL*4 OPTMC,PAR(1),TOL,DPAR(1) 
      EXTERNAL OPTMC
      LOGICAL*4 VERBOSE
      INTEGER*4 NEV
      COMMON /MCSETTING/ VERBOSE,NEV

           
1     FORMAT(1x,a4,' = ',F8.4,' [m-1] ',a10)
5     FORMAT(' Numerical optimization failed ! ') 
6     FORMAT(' Wrong syntax. Type> MRO n [e] ',/,
     *  'n=1 to 4 for ROMH, ROMV, ROAH, ROAV',/,
     *  'e   .. number of events in 1000 (default e=1)') 
                 
7     FORMAT(' (1) Incident flux',/,
     *       ' (2) flux/dE ',/,
     *       ' (3) flux/dE^2 ',/,
     *       ' (4) Powder peak (detector with Soller)',/,
     *       ' (5) Powder peak (position-sensitive detector) ',/,
     *       ' (6) Vanad peak ',/,
     * 'Select figure of merit: ',$)


20    WRITE(*,7)
      READ(SINP,*) I
      IF(I.LT.1.OR.I.GT.6) GOTO 20     
      OPTMERIT=I
c      WRITE(*,*) OPTMERIT
      
      CALL NESS_CONV(1)
      
      CALL GETROANAL(RO)  ! Analytical estimation    
      OPTPAR=NINT(RET(1))
      IF (NOS.GT.1) THEN
        OPTEV=RET(2)
      ELSE
        OPTEV=1.0        
      ENDIF        
      IF(OPTPAR.LT.0.OR.OPTPAR.GT.4.OR.NOS.LT.1) THEN
         WRITE(SOUT,6)
         RETURN
      ENDIF
      
      TOL=0.1
      DPAR(1)=0.05 ! minimum increment for vert. curvature = 0.05m^-1
      IF (OPTPAR.EQ.1.OR.OPTPAR.EQ.3) DPAR(1)=0.01 ! 0.01m^-1 for hor. curv.
      PAR(1)=RO(OPTPAR)
c      VERBOSE=.FALSE.
      CALL LMOPT(OPTMC,PAR,1,TOL,DPAR,0)
c      VERBOSE=.TRUE.
      IF (IERR.NE.0) THEN
         WRITE(SOUT,5)
         RETURN
      ENDIF         

      RES_DAT(i_ROMH+OPTPAR-1)=PAR(1)
      WRITE(SOUT,1) RES_NAM(i_ROMH+OPTPAR-1),RES_DAT(i_ROMH+OPTPAR-1),
     & ' changed'   
                            
      RETURN
      END
      
C
C
C
C***********************************************************************
      SUBROUTINE SAM_FLUX(ICOM)
C /// simulate flux at the sample (arg<>2) or at the detector (arg=2)
C /// by forward method (ICOM=1) or "from the sample" (ICOM=0)
C /// ICOM=2- monitor at position RET(1)      
C***********************************************************************
C
      IMPLICIT NONE
      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      integer*4 ICOM
              
C// monitor 
      IF (ICOM.EQ.2) THEN
        CALL NESS_CONV(1)
        IMONIT=NINT(RET(1))
        IF (IMONIT.LE.7) THEN 
           CALL NESS(7,0.D0)
        ELSE
           CALL NESS(6,0.D0)
        ENDIF   
        IMONIT=-1
        RETURN
      ENDIF
      
C// no monitor
      IMONIT=-1
      CALL NESS_CONV(1)
      IF (ICOM.EQ.3) THEN   ! TAS 
         CALL NESS(8,0.D0)
      ELSE IF (ICOM.EQ.4) THEN  ! PWD
         CALL NESS(9,0.D0)
      ELSE IF (ICOM.EQ.5) THEN  ! PWDS
         CALL NESS(10,0.D0)
C// FLUX   (ICOM=1) or NFLUX (ICOM=0) command:
      ELSE          
         IF (RET(1).EQ.2) THEN      ! powder - PSD
           CALL NESS(4+ICOM,0.D0)
         ELSE IF (RET(1).EQ.3) THEN !  TAS
           CALL NESS(1,0.D0) 
         ELSE IF (RET(1).EQ.4) THEN !  TAS forward
           FTAS=1
           CALL NESS(1,0.D0)
           FTAS=0 
         ELSE IF (RET(1).EQ.11) THEN !  double cryst. (bragg scattering)
           CALL NESS(11,0.D0) 
         ELSE                       ! flux at the sample
           CALL NESS(2+ICOM,0.D0)
         ENDIF
      ENDIF 
      RETURN
      END                   
        
C***********************************************************************
      SUBROUTINE SCAN_CHI(ICOM)
C Make a scan with monochromator cutting angle   
C arguments are STEP [deg], NSTEPS, [NEVENTS] 
C simulates powder diffration
C***********************************************************************
C
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INCLUDE 'trax.inc'
      
      integer*4 ICOM,NCHI,I
      real*4 EV,DCHI,CHI0
        
1     FORMAT('CHI = ',G12.6)      

      IF (NOS.GE.2) THEN
        CALL NESS_CONV(1)
        CHI0=HIMON
        DCHI=RET(1)
        NCHI=NINT(RET(2))
        EV=10.D0
        IF(NOS.GE.3) EV=RET(3) ! number of events
        IF (NCHI.GT.100) NCHI=100
        IF (NCHI.LT.1) NCHI=1
        DO I=1,NCHI
          HIMON=CHI0+(I-(NCHI+1)/2)*DCHI
          CALL NESS_CONV(0)         
          WRITE(SOUT,1) MON.CHI*180/PI
          CALL NESS(9,-ABS(EV))
        ENDDO
        HIMON=CHI0
        CALL NESS_CONV(0)         
      ENDIF  
      END

C***********************************************************************
      SUBROUTINE SCAN_TAS
C /// simulate standard TAS scan (DH,DK,DL,DE)
C//// using scattering cross-section defined by SQE_AMAG funciton
C accepts 4 arguments:
c a1 ... number of steps (obligatory) 
c a2 ... number of events (x1000) , default=10 
c a3 ... time (~monitor counts), default=100 
c a4 ... background (in cnts), default=0
C***********************************************************************
C
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INCLUDE 'rescal.inc'
        
      INTEGER*4 I,J,NSTP
      REAL*8 EV,MONEF
      PARAMETER(MONEF=1D-8)
            
      REAL*8 CNTS(128),CNTD(128),CNTE(128),KI(128),TIME,BCG
      REAL*8 QHKL0(4),K0     
      REAL*4 GASDEV
            
1     FORMAT(I3,2x,4(G10.4,1x),2(G12.4,2x))      
2     FORMAT('PNT   QH   QK   QL   EN   CNTS  MON')      
3     FORMAT('PNT   QH   QK   QL   EN   CNTS  TIME')      

      IF (NOS.GE.1) THEN
        K0=STP.KI
        CALL NESS_CONV(1)
        DO J=1,4
          QHKL0(J)=QHKL(J)
        ENDDO
c        WRITE(SOUT,1) (QHKL0(J),J=1,4)
c        pause
        NSTP=NINT(RET(1))
        EV=10.D0
        IF(NOS.GE.2) EV=RET(2) ! number of events
        TIME=100.D0
        IF(NOS.GE.3) TIME=RET(3) ! time
        BCG=0.D0
        IF(NOS.GE.4) BCG=RET(4) ! background
        IF (NSTP.GT.100) NSTP=100
        IF (NSTP.LT.1) NSTP=1
        WRITE(SOUT,2)
        DO I=1,NSTP
          DO J=1,4
            QHKL(J)=QHKL0(j)+(I-(NSTP+1)/2)*DELQ(J)
          ENDDO            
          CALL NESS_CONV(1)         
          CALL NESS(8,ABS(EV))
          CNTS(I)=IINC
          CNTD(I)=TIME*I3AX
          CNTE(I)=TIME*DI3AX
          KI(I)=STP.KI
          WRITE(SOUT,1) I,(QHKL(J),J=1,4),CNTD(I),CNTS(I) 
        ENDDO
        DO J=1,4
          QHKL(J)=QHKL0(J)
        ENDDO        
        CALL NESS_CONV(1)         
        DO I=1,NSTP
          SPCX(I)=QHKL0(4)+(I-(NSTP+1)/2)*DELQ(4)
          SPCY(I)=CNTD(I)/CNTS(I)/MONEF ! normalize to monitor counts
          SPCD(I)=CNTE(I)/CNTS(I)/MONEF
        ENDDO
        WRITE(SOUT,3)
        DO I=1,NSTP
          IF (BCG.GT.0) THEN ! add const. background and errors
            SPCY(I)=SPCY(I)+BCG
            SPCD(I)=SQRT(ABS(SPCY(I))+SPCD(I)**2)
            SPCY(I)=SPCY(I)+SQRT(ABS(SPCY(I)))*GASDEV()
          ENDIF  
          
          WRITE(SOUT,1) I,(QHKL0(j)+(I-(NSTP+1)/2)*DELQ(J),J=1,4),
     *                   SPCY(I),TIME/MONEF/CNTS(I)*KI(I)
        ENDDO
        SPCN=NSTP
      ENDIF  
      END

C***********************************************************************
      SUBROUTINE SCAN_THETA
C /// simulate standard TAS scan (A1,A2,A3,A4,A5,A6)
C accepts 4 arguments:
c a1 ... number of steps (obligatory) 
c a2 ... number of events (x1000) , default=10 
c a3 ... time (~monitor counts), default=100 
c a4 ... background (in cnts), default=0
C***********************************************************************
C
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
        
      INTEGER*4 MF
      PARAMETER (MF=65)
      INTEGER*4 NSTP
      REAL*8 EV,MONEF
      PARAMETER(MONEF=1D-8)
      
      INTEGER*4 I_IO
      CHARACTER*128 LINE
      character*50 filename
      REAL*8 AR(6)
      INTEGER*4 NA,IA(6),IWHAT,K,I,J
      REAL*4 FX(MF),FY(MF),DFY(MF),FY1(MF),DFY1(MF)
      REAL*8 CNTS(128),CNTD(128),CNTE(128),TIME
            
4     FORMAT('A',I1,'      ',$)      
5     FORMAT(G10.4,1x,$)
55    FORMAT(2(G10.4,1x))      
6     FORMAT('Axes [1..6]: ',$)
7     FORMAT(a)      
8     FORMAT('Steps [min]: ',$)
9     FORMAT('(1) Sample, (2) Powder, (3) Double-Crystal, (4) TAS :',$)
44     FORMAT(1x,4(2x,E13.5))

      IF (NOS.LT.1) THEN
        WRITE(SOUT,*) 'Use number of points as the 1st argument'
        RETURN
      ENDIF  
      WRITE(SOUT,9) 
      READ(SINP,*) IWHAT
      IF (IWHAT.EQ.4) THEN
        CALL SCAN_TAS
        RETURN
      ELSE IF (IWHAT.LT.1.OR.IWHAT.GT.4) THEN
        WRITE(*,*) 'UNDEFINED TASK: ',IWHAT 
        RETURN  
      ENDIF
C// initialize
        CALL NESS_CONV(1)
        DO J=1,6
          DTHAX(J)=0.
          IA(J)=J
          AR(J)=0.
        ENDDO
C// interpret arguments
        NSTP=NINT(RET(1))
        EV=10.D0
        IF(NOS.GE.2) EV=RET(2) ! number of events
        TIME=1.D0
        IF(NOS.GE.3) TIME=RET(3) ! time
        IF (NSTP.GT.101) NSTP=101
        IF (NSTP.LT.1) NSTP=1
C// read angular steps from input        
        WRITE(SOUT,6) 
        READ(SINP,7) LINE   ! read axes indexes
        CALL GETLINPARG(LINE,AR,6,NA)
        DO I=1,NA
          IA(I)=INT(AR(I))
          IF(IA(I).GT.6.OR.IA(I).LE.0) IA(I)=0
        ENDDO  
        WRITE(SOUT,8) 
        READ(SINP,7) LINE  ! read axes steps
        
        CALL GETLINPARG(LINE,AR,6,J) 
        IF (J.NE.NA) THEN
          WRITE(*,*) 'EACH AXIS MUST HAVE A STEP DEFINED !!'
        ENDIF
          DO I=1,NA
             WRITE(SOUT,4) IA(I) 
          ENDDO
          WRITE(SOUT,*)
          DO I=1,NA
             WRITE(SOUT,5) AR(I) 
          ENDDO
          WRITE(SOUT,*)
C// get only valid steps
        K=0
        DO I=1,NA
          IF (IA(I).NE.0.AND.AR(I).NE.0) THEN
             K=K+1
             AR(K)=AR(I)
             IA(K)=IA(I)
             WRITE(SOUT,4) IA(I)  ! write header
          ENDIF               
        ENDDO
        NA=K
        WRITE(SOUT,*) 'CNTS    ERR' 
        DO J=1,MF
          FY1(J)=0
          DFY1(J)=0            
        ENDDO

C// Start scan
        DO I=1,NSTP
          DO J=1,NA
            DTHAX(IA(J))=(I-(NSTP+1)/2)*AR(J) 
          ENDDO
          DO J=1,NA
            WRITE(SOUT,5) DTHAX(IA(J)) 
          ENDDO
          CALL NESS_CONV(0) 
          IF (IWHAT.EQ.1) THEN        
              CALL NESS(2,ABS(EV)) 
              CNTD(I)=TIME*IINC
              CNTE(I)=TIME*DIINC
          ELSE IF (IWHAT.EQ.2) THEN 
              CALL NESS(4,ABS(EV)) 
              CNTD(I)=TIME*IPWD
              CNTE(I)=TIME*DIPWD
          ELSE IF (IWHAT.EQ.3) THEN 
              CALL NESS(11,ABS(EV)) 
              CNTD(I)=TIME*I3AX
              CNTE(I)=TIME*DI3AX
          ENDIF              
          CNTS(I)=IINC
          WRITE(SOUT,55) CNTD(I),CNTE(I)
          CALL PSD_ARRAY(FX,FY,DFY,MF)
          DO J=1,MF
            FY1(J)=FY1(J)+FY(J)
            DFY1(J)=DFY1(J)+DFY(J)**2            
          ENDDO
        ENDDO
C// End scan, reset configuration
        DO J=1,MF
            DFY1(J)=SQRT(DFY1(J))            
        ENDDO
        DO J=1,6
          DTHAX(J)=0.
        ENDDO
        CALL NESS_CONV(0)   
C// save integrated profile at the PSD        
      I_IO=22
      filename=' '
12    format(a50)
13    FORMAT(' PSD data output: ',$)
      WRITE(SOUT,13)
      read(SINP,12) filename

      IF(filename(1:1).EQ.' '.OR.filename(1:1).EQ.CHAR(0)) then  ! generate automatic filename
        GOTO 200
      ELSE
        Open(Unit=i_IO,File=filename,err=999,Status='Unknown')
        write(i_io,*) 'X      INT       ERR    '
        do i=1,MF
          write(i_IO,44) FX(I),FY1(I),DFY1(I) 
        enddo  
        close(i_io)
      ENDIF
             
C// Fill arrays with results
200        DO I=1,NSTP
          SPCX(I)=(I-(NSTP+1)/2)*AR(1)
          SPCY(I)=CNTD(I)
          SPCD(I)=CNTE(I)
        ENDDO
C// List data
        DO J=1,NA
          WRITE(SOUT,4) J 
        ENDDO
        WRITE(SOUT,*) 'CNTS    ERR'         
        DO I=1,NSTP
          DO J=1,NA
            WRITE(SOUT,5) (I-(NSTP+1)/2)*AR(J) 
          ENDDO
          WRITE(SOUT,55) SPCY(I),SPCD(I)          
        ENDDO
        SPCN=NSTP
      return  
999   write(*,*) 'Cannot open file as unit ',i_IO      
      return
      END
C
C***********************************************************************
      SUBROUTINE BENCH
C /// simulate flux at the sample (arg<>2) or at the detector (arg=2)
C /// by forward method (ICOM=1) or "from the sample" (ICOM=0)      
C***********************************************************************
C
      IMPLICIT NONE

      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INTEGER*4 NB  

      NB=1000
      IMONIT=-1
      IF(NOS.GT.0) NB=NINT(1000*RET(1))
      CALL NESS_CONV(1)
      CALL NESS(2,0.D0)
      CALL NESS_BENCH(NB)
      
      RETURN
      END                   
C
C
C***********************************************************************
      SUBROUTINE ROCK(ICR)
C simulates rocking curve for monochromator (icr=1) or analyzer (icr=2)
C arguments: NEVENTS, NSTEPS, STEP [min]   
C saves results in rcurve.dat   
C***********************************************************************
C
      IMPLICIT NONE

      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INTEGER*4 ICR,NC,NTH,I  
      REAL*8 RTH(129),DTH,DIVH,DIVV

      NC=1000
      NTH=65
      DTH=2./180./60.*PI
      DIVH=0.
      DIVV=0.
      IF(NOS.GT.0) NC=NINT(1000*RET(1))
      IF(NOS.GT.1) NTH=NINT(RET(2))
      IF(NTH.LT.11) NTH=11
      IF(NTH.GT.129) NTH=129
      IF(NOS.GT.2) DTH=RET(3)/180/60*PI
      IF(NOS.GT.3) DIVH=RET(4)/180/60*PI
      IF(NOS.GT.4) DIVV=RET(5)/180/60*PI
      CALL NESS_CONV(1)
      CALL SPEC_INI(0,3)
      if (NOS.GT.0.AND.RET(1).EQ.0.) THEN
        CALL TEST_SYMMETRY(NC,NTH,DTH,DIVH,DIVV)      
      ELSE
        CALL NESS_ROCK(ICR,NC,NTH,DTH,RTH,DIVH,DIVV)
        OPEN(22,FILE='rcurve.dat',STATUS='unknown',ERR=100)
1       format(a)
2       format(2(E11.5,4x))
        write(22,1) 'theta[min]   r(theta)'
        DO I=1,NTH
          write(22,2) (-(NTH-1)/2.+I*1.)*DTH*180*60/PI,RTH(I)
        ENDDO      
100     close(22)
      ENDIF  
      
      RETURN
      END                   
C
C***********************************************************************
      SUBROUTINE TYPECFG
C /// print complete configuration of all components
C***********************************************************************
C
      IMPLICIT NONE

       INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INTEGER*4 N

      CALL NESS_CONV(1)
      IF(NOS.GT.0) THEN
         N=NINT(RET(1))
         IF (N.GT.9) THEN
           CALL SPEC_INI(0,8)            
         ELSE
           CALL SPEC_INI(0,3)          
         ENDIF
         IF(N.EQ.1) CALL SLIT_WRITE(SOUT,sou)
         IF(N.EQ.2) CALL BENDER_WRITE(SOUT,gdea)
         IF(N.EQ.3) CALL BENDER_WRITE(SOUT,guide)
         IF(N.EQ.4) CALL BENDER_WRITE(SOUT,sol1)
         IF(N.EQ.5) CALL CRYST_WRITE(SOUT,mon)
         IF(N.EQ.6) CALL BENDER_WRITE(SOUT,sol2a)
         IF(N.EQ.7) CALL BENDER_WRITE(SOUT,sol2)
         IF(N.EQ.8) CALL SLIT_WRITE(SOUT,sam)
         IF(N.EQ.9) CALL BENDER_WRITE(SOUT,sol3)
         IF(N.EQ.10) CALL CRYST_WRITE(SOUT,ana)
         IF(N.EQ.11) CALL BENDER_WRITE(SOUT,sol4)
         IF(N.EQ.12) CALL SLIT_WRITE(SOUT,det)
      ELSE
          CALL SPEC_INI(0,8)
          CALL WRITE_SETUP(SOUT,8)
      ENDIF
      
      RETURN
      END                   

C-----------------------------------------------------------        
      SUBROUTINE SET_DEVICE(SARG)
C set graphics device string for PGPLOT
C-----------------------------------------------------------      
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      CHARACTER*(*) SARG
      CHARACTER*60 DST
      INTEGER*4 I,PGBEGIN
1     FORMAT(' Graphics device (? for help) : ',$)
2     format(a)
C//      write(sout,*) DEVSTR

      write(sout,*) "present device: ",DEVSTR
      if (SARG.NE.' ') THEN
         DEVSTR=SARG
         write(sout,*) "new device: ",DEVSTR(1:I)         
      else 
         write(SOUT,1)
         read(SINP,2) DST
201      I=PGBEGIN(0,DST,1,1)
         IF (I.NE.1) THEN
           write(smes,*) "PGBEGIN error: ",I
           DST='?'
           GOTO 201
         END IF
         CALL PGQINF('DEV/TYPE',DEVSTR,I)
         write(smes,*) "new device: ",DEVSTR(1:I)         
         CALL PGEND  
         return
      endif
      end
      
C-----------------------------------------------------------        
      SUBROUTINE SETVAR(IVAR)
C    
C-----------------------------------------------------------      
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'source.inc'
      INTEGER*4 IVAR

1     FORMAT(' Source flux [1e14 n/s/cm^2] : ',F10.4)
2     FORMAT(' Source temperature [K] : ',F6.0)
      if (IVAR.EQ.1) then         
         IF(NOS.NE.0) SFLUX=RET(1)
         write(SOUT,1) SFLUX
      endif   
      if (IVAR.EQ.2) then         
         IF(NOS.NE.0) STEMP=RET(1)
         write(SOUT,2) STEMP
      endif   
      return
      end


C-------------------------------------------------------------
      SUBROUTINE SETCFG(SARG)
C Read configuration file
C IF IREAD>0, prepare also all calculated fields and run TRAX!
c-------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      CHARACTER*(*) SARG
      CHARACTER*128 NAME,FN,FRES
      INTEGER*4 IS,IL,IRES,ISC,ILC
      LOGICAL*4 ISDEFNAME,ISCREATED

1     FORMAT(' Configuration file [*.cfg]: ',$)
11    FORMAT(' Configuration file [',a,']: ',$)
2     FORMAT(A)
3     FORMAT('Cannot find configuration file ',a,/,
     *       '=> trying default: ',a)
4     FORMAT('Cannot find configuration file ',a,/,
     *       '=> trying the previous one: ',a)
5     FORMAT('Could not open any configuration file !!')
6     FORMAT('Could not find default cfg. file ',
     &       '=> creating one in current directory')
          
      ISCREATED=.FALSE.
C Get filename from dialog or from the argument SARG 
      CALL BOUNDS(SARG,IS,IL)
      CALL BOUNDS(CFGNAME,ISC,ILC)
      IF (IL.EQ.0) THEN
        IF (ILC.LE.0) THEN ! both SARG and CFGNAME are empty
          write(smes,1)
          read(sinp,2) NAME
        ELSE
          write(sout,11) CFGNAME(ISC:ISC+ILC-1) ! offer current CFGNAME as default
          read(sinp,2) NAME 
          IF (NAME.EQ.' ')  NAME=CFGNAME(ISC:ISC+ILC-1)
        ENDIF  
      ELSE
        NAME=SARG(IS:IS+IL-1)
      ENDIF

10    CALL BOUNDS(NAME,IS,IL)
      FN=NAME(IS:IS+IL-1)
      ISDEFNAME=(FN(1:IL).EQ.RESCAL_DEFNAME) ! is FN the default filename ?
      
C Add .cfg extension if NAME doesn't have one
      IF (IL.LE.4.OR.FN(IL-3:IL).NE.'.cfg') THEN 
        IF (128.GE.IL+4) THEN ! append .cfg if there is enough space
          FN=NAME(IS:IS+IL-1)//'.cfg' 
          IL=IL+4
        ENDIF
      ENDIF
      CALL CHECKRESFILE(FN,IRES,FRES,SILENT)
C file not found:
      IF (IRES.LE.0) THEN
        IF(ISDEFNAME) THEN  ! default not found => create one
           WRITE(SOUT,6)
           CALL WRITEDEFCFG
           ISCREATED=.TRUE.
           GOTO 10
        ELSE IF (ILC.GT.0) THEN  ! there is a previous filename => try it
           WRITE(SOUT,4) FN(1:IL),CFGNAME(ISC:ISC+ILC-1)
           NAME=CFGNAME(ISC:ISC+ILC-1)
           GOTO 10
        ELSE IF (.NOT.ISCREATED) THEN ! try the default                  
           WRITE(SOUT,3) ' ',RESCAL_DEFNAME
           NAME=RESCAL_DEFNAME
           GOTO 10
        ELSE  ! something is wrong - file was created but cannot read it !
           WRITE(SOUT,5) ! should not happen except the lack of write privileges or quota
           RETURN           
        ENDIF
      ENDIF

C note: CFGNAME=FN is without path, FRES is complete pathname
      CALL BOUNDS(FRES,IS,IL) 
      CFGNAME=FN
      
      CALL READCFG(FRES(IS:IS+IL-1))  ! read parameters from *.cfg
      END

C-----------------------------------------------------------------------
      SUBROUTINE SETPATH(SARG)
C select search path for data files      
C-----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      
      CHARACTER*(*) SARG
      CHARACTER*128 MYPATH
      INTEGER*4 IS,IL
      
1     FORMAT(' Path to data files [',a,'] : ',$)
2     FORMAT(A)       
3     FORMAT(' Data in ',a)
       
      CALL BOUNDS(DATPATH,IS,IL)
C Get pathname from dialog or from the argument SARG 
      IF (SARG.EQ.' ') THEN
        IF (IL.LE.0) THEN
          write(sout,1) 'current folder'
          read(sinp,2) MYPATH
        ELSE
          write(sout,1) DATPATH(IS:IS+IL-1)
          read(sinp,2) MYPATH 
          IF (MYPATH(1:1).EQ.' '.OR.MYPATH(1:1).EQ.CHAR(0)) THEN
             MYPATH=DATPATH(IS:IS+IL-1)
          ENDIF
        ENDIF  
      ELSE
        MYPATH=SARG
      ENDIF
C Interpret MYPATH, ensure that ending / is present       
      CALL BOUNDS(MYPATH,IS,IL)
      IF ((IL.LE.0).OR.
     *    (IL.EQ.1.AND.MYPATH(IS:IS+IL-1).EQ.'.').OR.
     *    (IL.EQ.2.AND.MYPATH(IS:IS+IL-1).EQ.'.'//PATHDEL)) THEN
         DATPATH=' '
         write(sout,3) 'current folder'
         RETURN
      ENDIF
      IF(MYPATH(IS+IL-1:IS+IL-1).NE.PATHDEL) THEN
         DATPATH=MYPATH(IS:IS+IL-1)//PATHDEL
      ELSE   
         DATPATH=MYPATH(IS:IS+IL-1)
      ENDIF
      write(sout,3) DATPATH(1:IL)
      END  
      
      
C-----------------------------------------------------------------------
      SUBROUTINE SETRESPATH(SARG)
C select search path for configuration files      
C-----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      
      CHARACTER*(*) SARG
      CHARACTER*128 MYPATH
      INTEGER*4 IS,IL
      
1     FORMAT(' Additional search path for configuration files [',
     *         a,'] : ',$)
2     FORMAT(A)       
3     FORMAT(' Configurations in ',a)
       
      CALL BOUNDS(RESPATH,IS,IL)
C Get pathname from dialog or from the argument SARG 
      IF (SARG.EQ.' ') THEN
        IF (IL.LE.0) THEN
          write(sout,1) 'current folder'
          read(sinp,2) MYPATH
        ELSE
          write(sout,1) RESPATH(IS:IS+IL-1)
          read(sinp,2) MYPATH 
          IF (MYPATH(1:1).EQ.' '.OR.MYPATH(1:1).EQ.CHAR(0)) THEN
             MYPATH=RESPATH(IS:IS+IL-1)
          ENDIF
        ENDIF  
      ELSE
        MYPATH=SARG
      ENDIF
C Interpret MYPATH, ensure that ending / is present       
      CALL BOUNDS(MYPATH,IS,IL)
      IF ((IL.LE.0).OR.
     *    (IL.EQ.1.AND.MYPATH(IS:IS+IL-1).EQ.'.').OR.
     *    (IL.EQ.2.AND.MYPATH(IS:IS+IL-1).EQ.'.'//PATHDEL)) THEN
         RESPATH=' '
         write(sout,3) 'current folder'
         RETURN
      ENDIF
      IF(MYPATH(IS+IL-1:IS+IL-1).NE.PATHDEL) THEN
         RESPATH=MYPATH(IS:IS+IL-1)//PATHDEL
      ELSE   
         RESPATH=MYPATH(IS:IS+IL-1)
      ENDIF
      write(sout,3) RESPATH(1:IL)
      END  

C-----------------------------------------------------------------------
      SUBROUTINE LIST
C-----------------------------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'rescal.inc'
      
      INTEGER*4 I1,I
       
19     FORMAT(' ',A5,' = ',G14.7,1X,$)                     
51     FORMAT(' ',2(A4,' = ',F10.5,1X),/,                   ! DM,DA
     2        ' ',3(A4,' = ',F10.2,1X),/,                   ! ETAM,ETAA,ETAS
     3        ' ',3(A4,' = ',F10.0,1X),/,                   ! SM,SA,SS 
     4        ' ',A4,' = ',F10.5,1X, A4,' = ',F10.0,1X,/,  ! KFIX,FX 
     5        ' ',4(A4,' = ',F10.2,1X),/,                   ! ALF1..4 
     6        ' ',4(A4,' = ',F10.2,1X),/,                   ! BET1..4
     7        4(' ',3(A4,' = ',F10.4,1X)/),                 ! AS,AA,AX,BX     
     1        ' ',4(A4,' = ',F10.4,1X),/,                   ! QH..EN
     2        ' ',4(A4,' = ',F10.4,1X),/,                   ! DQH..DE
     2        ' ',2(A4,' = ',F10.4,1X),/,                   ! DA3,DA4
     3        ' ',4(A4,' = ',F10.4,1X),/,                   ! GH..GL,GMOD
     4        ' ',4(A4,' = ',F10.4,1X),/,                   ! ROMH..ROAV
     5        ' ',2(A4,' = ',F10.2,1X))                     ! SDI,SHI

       IF (NOS.GE.1)   THEN  
         DO I=1,NOS    
           I1=NINT(RET(I))
           if (I1.GT.0.AND.I1.LT.RES_NVAR) THEN
              write(sout,19) RES_NAM(I1),RES_DAT(I1)
           endif
         ENDDO
         write(sout,*)
       ELSE
         write(sout,51) (RES_NAM(I),RES_DAT(I),I=1,RES_NVAR)   
c         nos=0
c         call SET_3AX(1)
c         call SET_3AX(3)
c         call SET_3AX(4)
c         call SET_3AX(5)
c         call SET_3AX(6)
       ENDIF
       END
       
C--------------------------------
      SUBROUTINE DOSHELL(COMM)
c execute shell command      
c--------------------------------
      implicit none
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      CHARACTER*(*) COMM
      CHARACTER*256 COMM1
      INTEGER*4 IS,L
      
1     FORMAT(' Command : ',$)
2     format(A)
      COMM1=' '
      if((COMM(1:1).EQ.' ').OR.(COMM(2:2).EQ.CHAR(0))) THEN 
        write(SOUT,1)
        read(SINP,2) COMM1
      else
        L=LEN(COMM)
        IF(L.GT.256) L=256
        COMM1=COMM(1:L)
      endif
      CALL BOUNDS(COMM1,IS,L)
      
      IF (L.GT.0) THEN
        CALL SYSTEM(COMM1(IS:IS+L-1))
      ENDIF
      end
      
***********************************************************************
      SUBROUTINE REINP(SARG)
C     redirection of input
C***********************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      character*(*) SARG
      INTEGER*4 IUINI,IUFILE,IRES
      DATA IUINI,IUFILE/0,10/
      
      IF(IUINI.EQ.0) IUINI=SINP      
      IF (SINP.NE.IUINI) CLOSE(SINP)
      if((SARG(1:1).ne.' ').and.(SARG(1:1).ne.char(0))) THEN
          CALL OPENRESFILE(SARG,IUFILE,IRES,0) 
          IF (IRES.LE.0) GOTO 2002        
          SINP=IUFILE
      else
          SINP=IUINI
      endif   
      RETURN     

2002  write(smes,*) 'Cannot open input file '//SARG
      SINP=IUINI
      END 

***********************************************************************
      SUBROUTINE REOUT(SARG)
C     redirection of input
C***********************************************************************
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      character*(*) SARG
      INTEGER*4 IUINI,IUFILE
      DATA IUINI,IUFILE/0,11/
      
      IF(IUINI.EQ.0) IUINI=SOUT      
      IF (SOUT.NE.IUINI) CLOSE(SOUT)
      if((SARG(1:1).ne.' ').and.(SARG(1:1).ne.char(0))) THEN
          OPEN (UNIT=IUFILE,ERR=2002,NAME=SARG, STATUS='UNKNOWN')          
          SOUT=IUFILE
      else
          SOUT=IUINI
      endif   
C      WRITE(*,*) SOUT
      RETURN     

2002  write(smes,*) 'Cannot open output file  '//SARG
      SOUT=IUINI
      END 

C-----------------------------------------------------
      SUBROUTINE READINIFILE(JOBNAME)
C   read initialization file
C CFGNAME = configuration file  
C DATAPATH = path to the data files   
C OPENFILE = data or RESCAL file to open   
C return JOBNAME .. filename of a job file to be executed at the startup 
c-----------------------------------------------------
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      CHARACTER*128 LINE
      CHARACTER*(*) JOBNAME
      INTEGER*4 IRES,IERR
      
1     FORMAT(A)
      JOBNAME=' '
      CALL OPENRESFILE('restrax.ini',22,IRES,0)
      IF(IRES.GT.0) THEN
         IRES=0
         DO WHILE(IRES.EQ.0)
           READ(22,1,END=100,iostat=IRES) LINE
           IF(LINE(1:1).NE.'#') THEN 
             CALL READ_STR('CFGNAME',LINE,CFGNAME,IERR)
             CALL READ_STR('DATAPATH',LINE,DATPATH,IERR)
             CALL READ_STR('JOB',LINE,JOBNAME,IERR)
             CALL READ_STR('OPENFILE',LINE,RESCAL_NAME,IERR)
           ENDIF  
         ENDDO
100      CLOSE(22)      
      ENDIF
      END
       

C----------------------------------------------------------------------------- 
      SUBROUTINE PROCARG
C// Process command line arguments for SIMRES
C----------------------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'structures.inc'
      INCLUDE 'ness_common.inc'
      INCLUDE 'randvars.inc'
      INCLUDE 'source.inc'
      INTEGER*4 I,J,M,IS,IL
      character*128 S
      INTEGER*4 IARGC
      
C Handle command-line options
      M=IARGC()
      
C Derive path to default setup files from executable pathname
      
      call GETARG(0,S)
      i=INDEX(S,PATHDEL//'bin'//PATHDEL)       
      IF (i.gt.1) then
         CFGPATH=S(1:i)//'setup'//PATHDEL
      ELSE
         CFGPATH='setup'//PATHDEL
      ENDIF
        
      idbg=0
      IRND=0
      IOPT=1  ! automatic optimization by default
      NORMMON=0 ! constant monitor efficiency (NOT ~ 1/k)
      MDIST=0
      M=IARGC()
      DO I=1,M
        call GETARG(I,S)
        if (S(1:5).eq.'-dir=') THEN
           CALL BOUNDS(S,IS,IL)
           IS=IS+5
           IL=IL-5
           IF(IL.GT.0) THEN
             RESPATH=S(IS:IS+IL-1)
             IF(RESPATH(1:IL).NE.PATHDEL) THEN ! add path delimiter
                RESPATH=RESPATH(1:IL)//PATHDEL
                IL=IL+1
             ENDIF
             write(SOUT,*) 'dir='//RESPATH(1:IL)
           ENDIF  
        else if (S(1:2).eq.'-d') then
           IDBG=2
           read(S(3:3),*,err=10) J
           if(J.NE.0) IDBG=J
        endif   
10      if (S(1:2).eq.'-s') THEN
           read(S(3:30),*) J
           if(J.NE.0) THEN 
               ISEED=ABS(J) 
               write(SOUT,*) 'SEED=',ISEED
           ENDIF
        endif
        if (S(1:2).eq.'-t') THEN
           read(S(3:30),*,err=20) J
           CALL RAN1SEED(ISEED)
           write(*,*) 'Test of the random number generator:'
           call RAN1TEST(J,1000000*J)
20         GOEND=1
           RETURN
        endif
        if (S(1:4).eq.'-flx') THEN
          CALL READ_FLUX(S(5:))
        endif
        if (S(1:4).eq.'-flh') THEN
           read(S(5:30),*,err=30) FLXH
           FLXH=FLXH*PI/180
        endif
30      if (S(1:4).eq.'-flv') THEN
           read(S(5:30),*,err=35) FLXV
           FLXV=FLXV*PI/180
        endif
35      if (S(1:3).eq.'-RB') THEN
           read(S(4:30),*,err=40) CBAR
           write(*,*) 'Right barrier [mm]: ',CBAR
        endif
40     if (S(1:3).eq.'-MX') THEN
           read(S(4:30),*,err=50) CMX
           write(*,*) 'crystal shift x [mm]: ',CMX
        endif
50      if (S(1:6).eq.'-Voigt') THEN
           MDIST=1
           write(*,*) 'pseudo-Voigt mosaic distribution'
        else if (S(1:7).eq.'-Lorenz') THEN
           MDIST=2
           write(*,*) 'Lorenzian mosaic distribution'
        else if (S(1:4).eq.'-Uni') THEN
           MDIST=3
           write(*,*) 'Uniform mosaic distribution'
        else 
           MDIST=0
           write(*,*) 'Gaussian mosaic distribution'
        endif
        if (S(1:4).eq.'-sil') THEN
           read(S(5:30),*,err=60) SILENT 
           write(SOUT,*) 'SILENT=',SILENT 
        endif
60      if (S(1:5).eq.'-ran1') THEN
           IRND=1
           write(*,*) 'Numerical Recipes RAN1 generator'
        endif
        if (S(1:5).eq.'-rand') THEN
           IRND=2
           write(*,*) 'System random number generator'
        endif
        if (S(1:6).eq.'-noopt') THEN
           IOPT=0
           write(*,*) 'No automatic sampling optimization'
        endif
        if (S(1:5).eq.'-nmon') THEN
           NORMMON=1
           write(*,*) 'Incident intensities ~ 1/ki'
        endif
        if (S(1:6).eq.'-cross') then
           ISQOM=0
           read(S(7:7),*,err=70) J
           if(J.GT.0.AND.J.LE.3) ISQOM=J
           if (ISQOM.EQ.1) then
             write(*,*) 'SCAN with antif. magnon cross-section'
           else if (ISQOM.EQ.2) then
             write(*,*) 'SCAN with Vanadium sample'
           endif  
        endif   
70      if (S(1:4).eq.'-log') THEN
           LOGFILE=S(5:30)
           write(*,*) 'Events logged in '//LOGFILE
100        format('Log events between [min max]: ',$)
           write(*,100)
           read(*,*) LOGMIN,LOGMAX
        endif
        if (S(1:5).eq.'-help'.or.S(1:1).eq.'?') THEN
           DO J=1,18
              write(*,*) HLPOPT(J)
           ENDDO
           GOEND=1            
        endif
        ENDDO   
      END
      
C----------------------------------------------------------------------------- 
        SUBROUTINE RESINIT
C// initialize RESTRAX
C// include all actions necessary to allocate memory, 
C// initialize variables, print LOGO etc..                
C----------------------------------------------------------------------------- 
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      INCLUDE 'config.inc'
      INCLUDE 'randvars.inc'
      INCLUDE 'rescal.inc'

      INTEGER*4 IRES
      character*128 extname,outname,FNAME,JOBNAME
      REAL*8 HMOS_DIST
      INTEGER*4 I,READ_MIRROR
      EXTERNAL HMOS_DIST
      
20    FORMAT('Using default RESCAL parameters: ',a)
      
C initialize error function
      CALL ERF_INIT(HMOS_DIST,-6.D+0,6.D+0)
C clear mirror and flux lookup tables
      I=READ_MIRROR(-1.D0)
      CALL READ_FLUX(' ')
      
C set path delimiter for M$ Windows
      CALL MKUPCASE(SYSNAME)
      IF (SYSNAME(1:7).EQ.'WINDOWS') THEN
        PATHDEL='\'
      ENDIF
      
C default silence level:
      SILENT=1
      
C initialize LINP
      CALL LINPSET(RES_NVAR+RES_NCMD,'SimRes',RES_NAM,RES_HLP)
      CALL LINPSETIO(SINP,SOUT,SMES)

      GOEND=0      
C Handle command-line options
      CALL PROCARG 
      IF (GOEND.NE.0) CALL RESEND
      
      CALL RAN1SEED(ISEED)       ! Initialize random number generator
      CALL LOGO                  ! print LOGO
      CALL READINIFILE(JOBNAME) ! read restrax.ini file
      CALL SETRESPATH(RESPATH)   ! set default path for configuration
      CALL UNITS(CUNIT)          ! set units for energy (meV)
      CALL SETPATH(DATPATH)      ! set data path to current dir.
      CALL SET_CRYST('Ge 111  ','Ge 111  ') ! read some crystal parameters
      CALL GETENV('PGPLOT_DEV',FNAME)
      IF(FNAME(1:1).NE.' ') THEN
         DEVSTR=FNAME
      ENDIF
      CALL OPENFILE(RESCAL_NAME,IRES)
      IF (IRES.LE.0) THEN      
         CALL SETDEFRES
         WRITE(SMES,20) RESCAL_DEFNAME
      ENDIF   
      CALL SETCFG(CFGNAME)       ! Read the configuration file
c job file required by restrax.ini      
      
      IF (JOBNAME(1:1).NE.' ') THEN 
         CALL REINP(JOBNAME)
         CALL LINPSETIO(SINP,SOUT,SMES)
         RETURN  
      ENDIF

C ask for a job file
2000  format(a30)
2001  format(' batch file  : ',$)
2004  format(' output file : ',$) 
      write(*,2001)   
      read(*,2000) extname
      if((extname(1:1).ne.' ').and.(extname(1:1).ne.char(0))) THEN
        CALL CHECKRESFILE(extname,IRES,FNAME,SILENT)
        IF (IRES.GT.0) THEN
           write(*,2004)
           read(*,2000) outname
           if((outname(1:1).ne.' ').and.(outname(1:1).ne.char(0))) THEN             
              CALL REOUT(outname)
              write(SOUT,*) 'RESTRAX - batch job '//extname
           endif
           CALL REINP(FNAME)    
           CALL LINPSETIO(SINP,SOUT,SMES)
c           write(*,*) 'input/output is ',SINP,'/',SOUT
        endif
      endif
           
      END

C-----------------------------------------------------------------------------   
        SUBROUTINE RESEND
C// end of RESTRAX
C// include all actions necessary to deallocate memory etc...                
C-----------------------------------------------------------------------------  
      IMPLICIT NONE
      INCLUDE 'const.inc'
      INCLUDE 'inout.inc'
      
      CALL REINP(' ')
      CALL REOUT(' ')
      CALL NESSEND   !  NESSEND must be called to deallocate
      WRITE(SMES,*) ' -> End of ResTrax'
      STOP 
      END
 
